// GoToSocial
// Copyright (C) GoToSocial Authors admin@gotosocial.org
// SPDX-License-Identifier: AGPL-3.0-or-later
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Affero General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Affero General Public License for more details.
//
// You should have received a copy of the GNU Affero General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

package customemojis

import (
	"net/http"

	apiutil "code.superseriousbusiness.org/gotosocial/internal/api/util"
	"code.superseriousbusiness.org/gotosocial/internal/config"
	"code.superseriousbusiness.org/gotosocial/internal/gtserror"
	"github.com/gin-gonic/gin"
)

// CustomEmojisGETHandler swagger:operation GET /api/v1/custom_emojis customEmojisGet
//
// Get an array of custom emojis available on the instance.
//
// If the instance config setting `instance-expose-custom-emojis` is `true` then authentication is not required.
//
//	---
//	tags:
//	- custom_emojis
//
//	produces:
//	- application/json
//
//	security:
//	- OAuth2 Bearer:
//		- read:custom_emojis
//
//	responses:
//		'200':
//			description: Array of custom emojis.
//			schema:
//				type: array
//				items:
//					"$ref": "#/definitions/emoji"
//		'401':
//			description: unauthorized
//		'406':
//			description: not acceptable
//		'500':
//			description: internal server error
func (m *Module) CustomEmojisGETHandler(c *gin.Context) {
	// If custom emojis are not exposed to unauthed
	// callers, fail if a token was not provided.
	if !config.GetInstanceExposeCustomEmojis() {
		if _, errWithCode := apiutil.TokenAuth(c,
			true, true, true, true,
			apiutil.ScopeReadCustomEmojis,
		); errWithCode != nil {
			apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
			return
		}
	}

	if _, err := apiutil.NegotiateAccept(c, apiutil.JSONAcceptHeaders...); err != nil {
		apiutil.ErrorHandler(c, gtserror.NewErrorNotAcceptable(err, err.Error()), m.processor.InstanceGetV1)
		return
	}

	emojis, errWithCode := m.processor.Media().GetCustomEmojis(c)
	if errWithCode != nil {
		apiutil.ErrorHandler(c, errWithCode, m.processor.InstanceGetV1)
		return
	}

	apiutil.JSON(c, http.StatusOK, emojis)
}
