/* cache-regs.h: MN10300 cache management registers
 *
 * Copyright (C) 2005 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#ifndef _ASM_CACHE_REGS_H
#define _ASM_CACHE_REGS_H

#include <asm/cpu-regs.h>

#define xCACHE_NWAYS		4	/* number of ways in caches */
#define xCACHE_NENTRIES		256	/* number of entries in each way */
#define xCACHE_LINESIZE		16	/* bytes per entry */
#define xCACHE_LINESHIFT	4	/* shift for bytes per entry */
#define xCACHE_WAYDISP		0x1000	/* displacement of one way from the next */

#ifndef __ASSEMBLY__
#define xCACHE_DISPARITY	(xCACHE_NENTRIES * xCACHE_LINESIZE)
#else
#define xCACHE_DISPARITY	xCACHE_NENTRIES * xCACHE_LINESIZE
#endif

/* data cache purge registers
 * - read from the register to unconditionally purge that cache line
 * - write address&0xffffff00 to conditionally purge that cache line
 *   - set LSB to request invalidation as well
 */
#define DCACHE_PURGE(WAY,ENTRY)	\
	__SYSREG(0xc8400000+(WAY)*xCACHE_WAYDISP+(ENTRY)*xCACHE_LINESIZE,u32)

#define DCACHE_PURGE_WAY0(ENTRY)	__SYSREG(0xc8400000+(ENTRY)*xCACHE_LINESIZE,u32)
#define DCACHE_PURGE_WAY1(ENTRY)	__SYSREG(0xc8401000+(ENTRY)*xCACHE_LINESIZE,u32)
#define DCACHE_PURGE_WAY2(ENTRY)	__SYSREG(0xc8402000+(ENTRY)*xCACHE_LINESIZE,u32)
#define DCACHE_PURGE_WAY3(ENTRY)	__SYSREG(0xc8403000+(ENTRY)*xCACHE_LINESIZE,u32)

/* instruction cache access registers */
#define ICACHE_DATA(WAY,ENTRY,OFF)	__SYSREG(0xc8000000+(WAY)*0x1000+(ENTRY)*0x10+(OFF)*4,u32)
#define ICACHE_TAG(WAY,ENTRY)		__SYSREG(0xc8100000+(WAY)*0x1000+(ENTRY)*0x10,u32)

/* instruction cache access registers */
#define DCACHE_DATA(WAY,ENTRY,OFF)	__SYSREG(0xc8200000+(WAY)*0x1000+(ENTRY)*0x10+(OFF)*4,u32)
#define DCACHE_TAG(WAY,ENTRY)		__SYSREG(0xc8300000+(WAY)*0x1000+(ENTRY)*0x10,u32)
#define xCACHE_TAG_VALID	0x00000001	/* cache tag valid bit */
#define xCACHE_TAG_DIRTY	0x00000008	/* data cache tag dirty bit */
#define xCACHE_TAG_ENTRY	0x00000ff0	/* cache tag entry address mask */
#define xCACHE_TAG_ADDRESS	0xfffff000	/* cache tag line address mask */

/*
 * primitive routines
 */
#ifndef __ASSEMBLY__
#ifndef CONFIG_MN10300_CACHE_DISABLED
extern void mn10300_icache_inv(void);
extern void mn10300_dcache_inv(void);
extern void mn10300_dcache_inv_page(unsigned start);
extern void mn10300_dcache_inv_range(unsigned start, unsigned end);
extern void mn10300_dcache_inv_range2(unsigned start, unsigned size);
#ifdef CONFIG_MN10300_CACHE_WBACK
extern void mn10300_dcache_flush(void);
extern void mn10300_dcache_flush_page(unsigned start);
extern void mn10300_dcache_flush_range(unsigned start, unsigned end);
extern void mn10300_dcache_flush_range2(unsigned start, unsigned size);
extern void mn10300_dcache_flush_inv(void);
extern void mn10300_dcache_flush_inv_page(unsigned start);
extern void mn10300_dcache_flush_inv_range(unsigned start, unsigned end);
extern void mn10300_dcache_flush_inv_range2(unsigned start, unsigned size);
#else
#define mn10300_dcache_flush()				do {} while(0)
#define mn10300_dcache_flush_page(start)		do {} while(0)
#define mn10300_dcache_flush_range(start, end)		do {} while(0)
#define mn10300_dcache_flush_range2(start, size)	do {} while(0)
#define mn10300_dcache_flush_inv()			mn10300_dcache_inv()
#define mn10300_dcache_flush_inv_page(start)		mn10300_dcache_inv_page(start)
#define mn10300_dcache_flush_inv_range(start, end)	mn10300_dcache_inv_range(start, end)
#define mn10300_dcache_flush_inv_range2(start, size)	mn10300_dcache_inv_range2(start, size)
#endif
#else
#define mn10300_icache_inv()				do {} while(0)
#define mn10300_dcache_inv()				do {} while(0)
#define mn10300_dcache_inv_page(start)			do {} while(0)
#define mn10300_dcache_inv_range(start, end)		do {} while(0)
#define mn10300_dcache_inv_range2(start, size)		do {} while(0)
#define mn10300_dcache_flush()				do {} while(0)
#define mn10300_dcache_flush_inv_page(start)		do {} while(0)
#define mn10300_dcache_flush_inv()			do {} while(0)
#define mn10300_dcache_flush_inv_range(start, end)	do {} while(0)
#define mn10300_dcache_flush_inv_range2(start, size)	do {} while(0)
#define mn10300_dcache_flush_page(start)		do {} while(0)
#define mn10300_dcache_flush_range(start, end)		do {} while(0)
#define mn10300_dcache_flush_range2(start, size)	do {} while(0)
#endif
#endif

#endif /* _ASM_CACHE_REGS_H */
