/* io.h: MN10300 I/O port emulation and memory-mapped I/O
 *
 * Copyright (C) 2005 Red Hat, Inc. All Rights Reserved.
 * Written by David Howells (dhowells@redhat.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */
#ifndef _ASM_IO_H
#define _ASM_IO_H

#include <asm/page.h> /* I/O is all done through memory accesses */
#include <asm/cpu-regs.h>

/*****************************************************************************/
/*
 * readX/writeX() are used to access memory mapped devices. On some
 * architectures the memory mapped IO stuff needs to be accessed
 * differently. On the x86 architecture, we just read/write the
 * memory location directly.
 */
static inline u8 readb(const volatile void __iomem *addr)
{
	return *(const volatile u8 *) addr;
}

static inline u16 readw(const volatile void __iomem *addr)
{
	return *(const volatile u16 *) addr;
}

static inline u32 readl(const volatile void __iomem *addr)
{
	return *(const volatile u32 *) addr;
}

#define __raw_readb readb
#define __raw_readw readw
#define __raw_readl readl

static inline void writeb(unsigned char b, volatile void __iomem *addr)
{
	*(volatile u8 *) addr = b;
}

static inline void writew(unsigned short b, volatile void __iomem *addr)
{
	*(volatile u16 *) addr = b;
}

static inline void writel(unsigned int b, volatile void __iomem *addr)
{
	*(volatile u32 *) addr = b;
}

#define __raw_writeb writeb
#define __raw_writew writew
#define __raw_writel writel

/*****************************************************************************/
/*
 * traditional input/output functions
 */
static inline unsigned long inb_local(unsigned long addr)
{
	return readb((volatile void __iomem *) addr);
}

static inline void outb_local(unsigned char b, unsigned long addr)
{
	return writeb(b, (volatile void __iomem *) addr);
}

static inline unsigned long inb(unsigned long addr)
{
	return readb((volatile void __iomem *) addr);
}

static inline unsigned long inw(unsigned long addr)
{
	return readw((volatile void __iomem *) addr);
}

static inline unsigned long inl(unsigned long addr)
{
	return readl((volatile void __iomem *) addr);
}

static inline void outb(unsigned char b, unsigned long addr)
{
	return writeb(b, (volatile void __iomem *) addr);
}

static inline void outw(unsigned short b, unsigned long addr)
{
	return writew(b, (volatile void __iomem *) addr);
}

static inline void outl(unsigned int b, unsigned long addr)
{
	return writel(b, (volatile void __iomem *) addr);
}

#define inb_p	inb
#define outb_p	outb

#define inw_p	inw
#define outw_p	outw


static inline void insb(unsigned long addr, void *buffer, int count)
{
	unsigned char *buf = buffer;
	while (count--)
		*buf++ = inb(addr);
}

static inline void insw(unsigned long addr, void *buffer, int count)
{
	unsigned short *buf = buffer;
	while (count--)
		*buf++ = inw(addr);
}

static inline void insl(unsigned long addr, void *buffer, int count)
{
	unsigned long *buf = buffer;
	while (count--)
		*buf++ = inl(addr);
}

static inline void outsb(unsigned long addr, const void *buffer, int count)
{
	const unsigned char *buf = buffer;
	while (count--)
		outb(*buf++, addr);
}

static inline void outsw(unsigned long addr, const void *buffer, int count)
{
	const unsigned short *buf = buffer;
	while (count--)
		outw(*buf++, addr);
}

extern void __outsl(unsigned long addr, const void *buffer, int count);
static inline void outsl(unsigned long addr, const void *buffer, int count)
{
	const unsigned long *buf;
	if ((long) buffer & 0x3)
		return __outsl(addr, buffer, count);
	buf = buffer;
	while (count--)
		outl(*buf++, addr);
}

#define IO_SPACE_LIMIT 0xffffffff

#ifdef __KERNEL__

#include <linux/vmalloc.h>
#define __io_virt(x) ((void *) (x))

/*
 * Change virtual addresses to physical addresses and vv.
 * These are pretty trivial
 */
static inline unsigned long virt_to_phys(volatile void * address)
{
	return __pa(address);
}

static inline void * phys_to_virt(unsigned long address)
{
	return __va(address);
}

/*
 * Change "struct page" to physical address.
 */
static inline void *__ioremap(unsigned long offset, unsigned long size, unsigned long flags)
{
	return (void *) offset;
}

static inline void *ioremap(unsigned long offset, unsigned long size)
{
	return (void *) offset;
}

/*
 * This one maps high address device memory and turns off caching for that area.
 * it's useful if some control registers are in such an area and write combining
 * or read caching is not desirable:
 */
static inline void *ioremap_nocache (unsigned long offset, unsigned long size)
{
        return (void *) (offset | 0x20000000);
}

static inline void iounmap(void *addr)
{
}

/*
 * PCI bus iomem addresses must be in the region 0x80000000-0x9fffffff
 */
static inline unsigned long virt_to_bus(volatile void * address)
{
#if 0
	extern const char _init_begin[0];
	if ((u32) address < (u32) _init_begin || (u32) address > 0x93FFFFFFU)
		printk("### VIRT_TO_BUS(%p,pc=%p)\n", address, current_text_addr());
#endif
	return ((unsigned long) address) & ~0x20000000;
}

static inline void *bus_to_virt(unsigned long address)
{
	return (void *) address;
}

#define page_to_bus page_to_phys

#define memset_io(a,b,c)	memset(__io_virt(a), (b), (c))
#define memcpy_fromio(a,b,c)	memcpy((a), __io_virt(b), (c))
#define memcpy_toio(a,b,c)	memcpy(__io_virt(a), (b), (c))


/*
 * Cache management for DMA
 */
#define dma_cache_inv(start, size) \
	mn10300_dcache_inv_range2(virt_to_phys(start), size))

#define dma_cache_wback(start, size) \
	mn10300_dcache_flush_range2(virt_to_phys(start), size))

#define dma_cache_wback_inv(_start,_size) \
	mn10300_dcache_flush_inv_range2(virt_to_phys(start), size))

#define flush_write_buffers() \
	mn10300_dcache_flush_inv()

#endif /* __KERNEL__ */

#endif /* _ASM_IO_H */
