/*
**
**	idtfpip.c   Floating Point Implementation Package
**
**	Copyright 1990 Integrated Device Technology, Inc.
**
*/
#include "excepthdr.h" 
#include "idtcpu.h"  
#include "fpip.h"


int fp_init();			/* initialization requirements               */
int fpget_fpcsr();		/* ret real|simulated fp control/status reg  */
int fpset_fpcsr();		/* set real|simulated fp control/status reg  */
int fpget_RM();			/* return rounding mode bits from fp csr     */
int fpset_RM();			/* set selected rounding mode in fp csr      */
int fpget_stickyBits();		/* return 5 Sticky Bits from fp csr          */
int fpclr_stickyBits();	        /* clear selected (or ALL) Sticky Bits       */
int fpset_stickyBits();		/* set selected Sticky Bits	             */	
int fp_enableTrap();		/* enable Trap for selected exceptions       */
int fp_disableTrap();		/* disable Trap for selected exceptions      */
int fp_signal(); 		/* set Exception Trap & define Handler       */
int fp_defaultHdlr();		/* IDT Default FP Exception Handling Routine */
void fp_int();			/* FP Exception Handling Switch		     */ 
void fpget_excregs();		/* fill the structure with register set      */
void fpset_excregs();		/* set the exc regs save buffer with args    */

struct exc_regs fpexc;		/* "real" exception registers here	     */	
int SWexc_fpcsr;		/* fpcsr at time of SW Int exception   	     */		
/***************************************************************************/

int ext_int_msk [] =  {
		0x0100,	/* external software interrupt 0 ~ Level 0 */
		0x0200,	/* external software interrupt 1 ~ Level 1 */
		0x0400,	/* external hardware interrupt 0 ~ Level 2 */
		0x0800,	/* external hardware interrupt 1 ~ Level 3 */
		0x1000,	/* external hardware interrupt 2 ~ Level 4 */
		0x2000,	/* external hardware interrupt 3 ~ Level 5 */
		0x4000,	/* external hardware interrupt 4 ~ Level 6 */
		0x8000	/* external hardware interrupt 5 ~ Level 7 */
		};

int (*IDTfpeh[6]) () = { 
			   fp_defaultHdlr,             /* INEXACT           */
			   fp_defaultHdlr,	       /* UNDERFLOW         */
			   fp_defaultHdlr,             /* OVERFLOW          */
			   fp_defaultHdlr,             /* DIVIDE by ZERO    */
			   fp_defaultHdlr,	       /* INVALID OPERAND   */
			   fp_defaultHdlr	       /* UNIMPLEMENTED     */
			};
	   		
/*-----------------------------------------------------------------------
**
**	fp_init(ext_int_level) - sets up the pointer to the floating point 
**                       exception handler routine.
**
**		Recognizes R3010 by HDW EXT_INTn_LN and Emulation mode by 
**		    Sw_Intn_Ln arguments
**	
**		ext_int_level - defined in excepthdr.h (SW/EXT_INTn_LN)
**
**		enable external hardware interrupt in CP0 Status Register
**
**		if Hardware interrupt
**		  set TrapEnable bits in FP control/status register OFF
**		elseif Software interrupt
**		  set TrapEnable bits in FP_CSR in fpexc structure OFF
**
*/ 
int fp_init(ext_int_level)
int ext_int_level;

{
	  add_ext_int_func(ext_int_level, fp_int);   /* FP Interrupt Handler */
	  if (ext_int_level > SW_INT1_LN)
     	    set_fpcsr(0);		             /* Disable all FP Traps */
 	  else
	    fpexc.FPCSR_Reg = 0;	             /* Disable all Em Traps */
          enable_int(ext_int_msk[ext_int_level]);    /* EN Status Reg Int Msk */
}

/*
**  FP Interrupt Handler - 
**			   Default - fp_defaultHdler unless replaced by
**				     user via fp_signal()
**          
**	  Save CP0 Exception Registers (CAUSE, STATUS) in fpexc structure
**	  if Hardware interrupt (R3010 present)
**	    read FPU Control/Status Register into fpexc structure	  
**	    read EPC into fpexc structure
**
**	  Switch to correct handler for each type of interrupt
**	  Turn Exception (Cause) bits in fpexc structure OFF
**	  if Hardware interrupt (R3010 present)
**	    replace FPU Control/Status Register with structure
*/

void fp_int()
{
  int hndlr;

	set_fp_flag();			/* tell excp hndlr this is fp int */

	fpexc.CAUSE_Reg = get_CAUSE();	  /* save all exception registers */
	fpexc.STATUS_Reg = get_STATUS();
	if (fpexc.STATUS_Reg & SR_IMASK2)  /* HDW only R3010 MUST be there */
	  {
 	    fpexc.FPCSR_Reg = get_fpcsr(); 	
	    fpexc.EPC_Reg = get_cp0epc();
	  }
	hndlr = ((fpexc.FPCSR_Reg & AllExceptions) >> 12); /* isolate cause*/
	switch (hndlr) {
	    case INEXACT:
                               IDTfpeh[INEXACTsig](hndlr); /* select handler */	
			       break; 

	    case UNDERFLOW:
                               IDTfpeh[UNDERFLsig](hndlr); /* select handler */	
				break; 

	    case OVERFLOW:
                               IDTfpeh[OVERFLsig](hndlr); /* select handler */	
			       break; 

	    case ZERODIV:
                                IDTfpeh[ZERODIVsig](hndlr); /* select handler*/	
				break; 

	    case INVALID:
                                IDTfpeh[INVALIDsig](hndlr); /* select handler*/	
				break; 

	    case UNIMPLEMENTED:
                                IDTfpeh[UNIMPLsig](hndlr); /* select handler*/	
				break; 
		}

        fpset_fpcsr(fpexc.FPCSR_Reg & ~AllExceptions); /* reset fpcsr  */
	clr_CAUSE();			   /* turn OFF any SW interrupts   */
}

/*
** fp_defaultHdlr - Floating Point Exception occurred and corresponding 
**                  TrapEnabled. IDT default handler.  (User may choose 
**                  alternate handler or NULL to ignore exception.
**          
**	Stand Alone Version:
**	    Do not do any internal , but leave template in place
**	    for more extensive user diagnostics
**	
**	USER:
**		Use this format to insert any exception handling
**		(in addition to getting the Exception  
**		Registers); e.g. replace operands, clear registers.
**
*/
int fp_defaultHdlr(exc)
int exc;
{
   /* 	printf("Reached IDT Default Floating Point Exception Handler:\n"); */
}


/*
** fpget_fpcsr - returns contents of FP  Control/Status Register  
**
**		if Hardware interrupt
**		  read FP control/status register directly
**		elseif Software interrupt
**		  get from  FP_CSR in fpexc structure
**
*/
int fpget_fpcsr()
{
	fpexc.STATUS_Reg = get_STATUS();
	if (fpexc.STATUS_Reg & SR_IMASK2)  /* HDW only R3010 MUST be there */
	  fpexc.FPCSR_Reg = get_fpcsr();   /* get current fpcsr */
        return(fpexc.FPCSR_Reg);   	   /* return real or simulated csr */
}


/*
** fpset_fpcsr - set fp Control/Status Register
**
**		if Hardware interrupt
**		  set FP control/status register in FPA
**		elseif Software interrupt
**		  set FP_CSR in fpexc structure 
**
*/
int fpset_fpcsr(new_csr)
int new_csr;
{
	fpexc.STATUS_Reg = get_STATUS();
	if (fpexc.STATUS_Reg & SR_IMASK2)  /* HDW only R3010 MUST be there */
	   set_fpcsr( new_csr );  	   /* set FP Control Status REg    */
	else
           fpexc.FPCSR_Reg = new_csr;      /* set simulated fpcsr          */
}

/*
** fpget_RM - returns Rounding Mode from fp Control/Status Register  
**
*/
int fpget_RM()
{
	fpexc.FPCSR_Reg = fpget_fpcsr();   /* get current fpcsr */
        return(fpexc.FPCSR_Reg & fpRM);    /* dump all but RM bits */
}


/*
** fpset_RM - set selected rounding mode in fp Control/Status Register
**
*/
int fpset_RM(RMode)
int RMode;
{
	fpset_fpcsr( (fpget_fpcsr()& ~fpRM) | RMode);  /* set rounding mode */
}


/*
** fpget_stickyBits - returns Sticky Bits from fp Control/Status Register  
**
*/
int fpget_stickyBits()
{
	fpexc.FPCSR_Reg = fpget_fpcsr();   /* get current fpcsr */
        return(fpexc.FPCSR_Reg & AllSticky >> 2); /* dump all but Sticky bits */
}


/*
** fpclr_stickyBits - clears selected Sticky Bits in fp Control/Status Register
**
*/
int fpclr_stickyBits(sticky)
int sticky;
{
	fpset_fpcsr(fpget_fpcsr()& ~sticky);    /* clear only sticky bits */
}



/*
** fpset_stickyBits - sets selected Sticky Bits in fp Control/Status Register
**
*/
int fpset_stickyBits(sticky)
int sticky;
{
	fpset_fpcsr(fpget_fpcsr()| sticky);    /* set only sticky bits */
}


/*
** fp_signal - provides a technique to replace the default exception handling
**             routine with user's routine or NULL to eliminate any handling
**	     - enables corresponding Trap for selected exception
**
**             NOTE: only one [or ALL] exceptions may be selected by fp_signal
*/
int fp_signal(fpTrapEn, fpeh)
int fpTrapEn;				/* specific exception selected */
int (*fpeh)();				/* exception handler or NULL   */

{
        switch (fpTrapEn)  {
             case TrINEXACT:
				IDTfpeh[INEXACTsig] = fpeh;
				break;
             case TrUNDERFLOW:
				IDTfpeh[UNDERFLsig] = fpeh;
				break;
	     case TrOVERFLOW:
				IDTfpeh[OVERFLsig] = fpeh;
				break;
	     case TrZERODIV:
				IDTfpeh[ZERODIVsig] = fpeh;
				break;
	     case TrINVALID:
				IDTfpeh[INVALIDsig] = fpeh;
				break;
	     case EnableAll:
	                        IDTfpeh[INEXACTsig] = fpeh;
				IDTfpeh[UNDERFLsig] = fpeh;
				IDTfpeh[OVERFLsig]  = fpeh;
				IDTfpeh[ZERODIVsig] = fpeh;
				IDTfpeh[INVALIDsig] = fpeh;
				break;
	     default:
				return(0);		
	       }
           
	fp_enableTrap(fpTrapEn);          /* set Trap Enable bit in fpcsr   */
	return(1);
}


/*
** fp_enableTrap - sets selected Trap Enable bits in fp Control/Status register 
**
**		captures Status Register for R3010 testing
**		if Hardware interrupt
**		  set TrapEnable bits in FP control/status register ON
**		elseif Software interrupt
**		  set TrapEnable bits in FP_CSR in fpexc structure ON
**
*/
int fp_enableTrap(TrapMsk)
int TrapMsk;
{
	fpexc.STATUS_Reg = get_STATUS();
	if (fpexc.STATUS_Reg & SR_IMASK2)  /* HDW only R3010 MUST be there */
	  set_fpcsr(get_fpcsr()|TrapMsk);  /* current fpcsr|Traps to be set */
        else
          fpexc.FPCSR_Reg |= TrapMsk;	   /* set Traps in SW version      */
}





/*
** fp_DisableTrap - clears Trap Enable bits in fp Control/Status register 
**
**		if Hardware interrupt
**		  set TrapEnable bits in FP control/status register OFF
**		elseif Software interrupt
**		  set TrapEnable bits in FP_CSR in fpexc structure OFF
**
*/
int fp_disableTrap(TrapMsk)
int TrapMsk;
{
	if (fpexc.STATUS_Reg & SR_IMASK2)  /* HDW only R3010 MUST be there */
	  set_fpcsr(get_fpcsr() & ~TrapMsk);  /* turn off selected Traps   */
        else
          fpexc.FPCSR_Reg &= ~TrapMsk;	   /* turn off traps in SW version */
}


/***************************************************************************
**
**      fpget_excregs(ptr) - fill the buffer with the exception registers
**			   as saved during the most recent fp interrupt.
**
**************************************************************************/

void fpget_excregs(regptr)
struct exc_regs *regptr;
{
	regptr->EPC_Reg = fpexc.EPC_Reg;
	regptr->CAUSE_Reg = fpexc.CAUSE_Reg;
	regptr->STATUS_Reg = fpexc.STATUS_Reg;
	regptr->FPCSR_Reg = fpexc.FPCSR_Reg;
	regptr->sav_FPCSR = fpexc.sav_FPCSR;
}


/***************************************************************************
**
**	fpset_excregs(ptr) - reset exception register save buffer 
**
**************************************************************************/

void fpset_excregs(regptr)
struct exc_regs *regptr;
{
	fpexc.EPC_Reg = regptr->EPC_Reg;
	fpexc.CAUSE_Reg = regptr->CAUSE_Reg;
	fpexc.STATUS_Reg = regptr->STATUS_Reg;
	fpexc.FPCSR_Reg = regptr->FPCSR_Reg;
	fpexc.sav_FPCSR = regptr->sav_FPCSR;
}
