﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NLog;
using System.Runtime.Serialization.Formatters.Binary;
using System.IO;

namespace Infinispan.HotRod
{
    /// <summary>
    /// Hotrod is a binary protocol so the object passed in by the user are
    /// being marshalled into a binary format using an implementation of this
    /// interface. Users can implement this interface and develop their own
    /// marshaller without using the default marshaller.
    /// </summary>
    public interface IMarshaller
    {
        /// <summary>
        /// Marshalls an object to a byte array. The estimatedSize parameter
        /// is a hint that can be passed in to allow for efficient sizing of
        /// the byte array before attempting to marshall the object.  The more
        /// accurate this estimate is, the less likely byte[]s will need to be
        /// resized to hold the byte stream generated by marshalling the object.
        /// </summary>
        /// <param name="obj">
        /// object to convert to a byte array. Must not be null.
        /// </param>
        /// <param name="estimatedSize">
        /// an estimate of how large the resulting byte array may be
        /// </param>
        /// <returns>
        /// a byte array with the marshalled form of the object
        /// </returns>
        byte[] ObjectToByteBuffer(Object obj, int estimatedSize);

        /// <summary>
        /// Marshalls an object to a byte array.
        /// </summary>
        /// <param name="obj">
        /// object to convert to a byte array.  Must not be null.
        /// </param>
        /// <returns>
        /// a byte array with the marshalled form of the object
        /// </returns>
        byte[] ObjectToByteBuffer(Object obj);

        /// <summary>
        /// Unmarshalls an object from a byte array.
        /// </summary>
        /// <param name="buf">
        /// byte array containing the binary representation of an object. Must
        /// not be null.
        /// </param>
        /// <returns>
        /// an object
        /// </returns>
        Object ObjectFromByteBuffer(byte[] buf);

        /// <summary>
        /// Unmarshalls an object from a specific portion of a byte array.
        /// </summary>
        /// <param name="buf">
        /// byte array containing the binary representation of an object. Must
        /// not be null.
        /// </param>
        /// <param name="offset">
        /// point in buffer to start reading
        /// </param>
        /// <param name="length">
        /// number of bytes to consider
        /// </param>
        /// <returns>
        /// an object
        /// </returns>
        Object ObjectFromByteBuffer(byte[] buf, int offset, int length);

        /// <summary>
        /// A method that checks whether the given object is marshallable as per
        /// the rules of this marshaller.
        /// </summary>
        /// <param name="o">
        /// object to verify whether it's marshallable or not
        /// </param>
        /// <returns>
        /// true if the object is marshallable, otherwise false
        /// </returns>
        bool IsMarshallable(Object o);
    }
}
