package org.jbpm.gd.jpdl.ui.properties;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.forms.widgets.ScrolledPageBook;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetPage;
import org.jbpm.gd.common.notation.AbstractNotationElement;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.CancelTimer;
import org.jbpm.gd.jpdl.model.CreateTimer;
import org.jbpm.gd.jpdl.model.Event;
import org.jbpm.gd.jpdl.model.EventContainer;
import org.jbpm.gd.jpdl.model.JpdlElement;
import org.jbpm.gd.jpdl.model.Mail;
import org.jbpm.gd.jpdl.model.Script;
import org.jbpm.gd.jpdl.ui.JpdlLabelProvider;
import org.jbpm.gd.jpdl.ui.graph.part.NotationElementGraphicalEditPart;
import org.jbpm.gd.jpdl.ui.outline.part.JpdlOutlineEditPart;


public class EventContainerSection extends AbstractPropertySection implements PropertyChangeListener {
	
    private TabbedPropertySheetPage tabbedPropertySheetPage;	
	private Tree eventTree;
	private EventContainer eventContainer;
    private EventContainerSectionActionBarContributor actionBarContributor;
    private ScrolledPageBook detailsArea;
    
    private EventConfigurationComposite eventConfigurationComposite;
    private ActionConfigurationComposite actionConfigurationComposite;
    private ScriptConfigurationComposite scriptConfigurationComposite;
    private CreateTimerConfigurationComposite createTimerConfigurationComposite;
    private CancelTimerConfigurationComposite cancelTimerConfigurationComposite;
    private MailConfigurationComposite mailConfigurationComposite;
    
    private Event selectedEvent;
    private ActionElement selectedActionElement;
	 	
	public void createControls(Composite parent,
			TabbedPropertySheetPage aTabbedPropertySheetPage) {
		super.createControls(parent, aTabbedPropertySheetPage);	
		actionBarContributor = new EventContainerSectionActionBarContributor(this);
		tabbedPropertySheetPage = aTabbedPropertySheetPage;
		final Composite composite = getWidgetFactory().createFlatFormComposite(parent);		
		createMasterArea(composite);
		createDetailsArea(composite);
	}
	
	private void createMasterArea(Composite composite) {
		eventTree = getWidgetFactory().createTree(
				composite, SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.FULL_SELECTION);
		eventTree.setLayoutData(createEventTreeLayoutData());
		eventTree.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleEventTreeSelection();
			}			
		});
		actionBarContributor.createPopupMenu(eventTree);
	}
	
	private void handleEventTreeSelection() {
		TreeItem[] selectedItems = eventTree.getSelection();
		if (selectedItems.length == 0) {
			unSelectAll();
		} else {
			JpdlElement element = (JpdlElement)selectedItems[0].getData();
			if (element instanceof Event) {
				selectEvent((Event)element);
			} else if (element instanceof ActionElement) {
				selectActionElement((ActionElement)element);
			}
		}
		updateDetailsArea();
	}
	
	private boolean somethingSelected() {
		return getSelectedEvent() != null  || getSelectedActionElement() != null;
	}
	
	private void updateDetailsArea() {
		if (getSelectedActionElement() != null) {
			detailsArea.showPage(getSelectedActionElement().getElementId());
		} else if (getSelectedEvent() != null) {
			detailsArea.showPage(getSelectedEvent().getElementId());
		} else {
			detailsArea.showEmptyPage();
		}
	}
	
	private void unSelectAll() {
		if (somethingSelected()) {
			selectActionElement(null);
		}
	}
	
	private void selectEvent(Event event) {
		if (event == getSelectedEvent() && selectedActionElement == null) return;
		actionBarContributor.setAddActionElementEnabled(event != null);
		actionBarContributor.setRemoveEnabled(event != null);
		if (selectedActionElement != null) {
			selectedActionElement.removePropertyChangeListener(this);
		}
		if (selectedEvent != null) {
			selectedEvent.removePropertyChangeListener(this);
		}
		selectedActionElement = null;
		selectedEvent = event;
		eventConfigurationComposite.setEvent(event);
		if (selectedEvent != null) {
			selectedEvent.addPropertyChangeListener(this);
		}
	}
	
	private void selectActionElement(ActionElement actionElement) {
		if (actionElement == getSelectedActionElement()) return;
		Event event = null;
		if (actionElement != null) {
			event = (Event)eventTree.getSelection()[0].getParentItem().getData();
		}
		selectEvent(event);
		selectedActionElement = actionElement;
		updateActionElementDetails();
		if (selectedActionElement != null) {
			selectedActionElement.addPropertyChangeListener(this);
		}
	}
	
	private void updateActionElementDetails() {
		if (selectedActionElement instanceof Action) {
			actionConfigurationComposite.setAction((Action)selectedActionElement);
		} else if (selectedActionElement instanceof Script) {
			scriptConfigurationComposite.setScript((Script)selectedActionElement);
		} else if (selectedActionElement instanceof CreateTimer) {
			createTimerConfigurationComposite.setCreateTimer((CreateTimer)selectedActionElement);
		} else if (selectedActionElement instanceof CancelTimer) {
			cancelTimerConfigurationComposite.setCancelTimer((CancelTimer)selectedActionElement);
		} else if (selectedActionElement instanceof Mail) {
			mailConfigurationComposite.setMail((Mail)selectedActionElement);
		}
	}
	
	private FormData createEventTreeLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, 5);
		data.right = new FormAttachment(20, -5);
		data.top = new FormAttachment(0, 5);
		data.bottom = new FormAttachment(100, -5);
		return data;
	}
	
	private void createDetailsArea(Composite composite) {
		detailsArea = getWidgetFactory().createPageBook(composite, SWT.V_SCROLL);
		detailsArea.setLayoutData(createDetailsAreaLayoutData());
		createEventPage();
		createActionPage();
		createScriptPage();
		createCreateTimerPage();
		createCancelTimerPage();
		createMailPage();
	}
	
	private void createActionPage() {
		Composite actionPage = detailsArea.createPage("org.jbpm.gd.jpdl.core.action");
		actionPage.setLayout(new FormLayout());
		actionConfigurationComposite = ActionConfigurationComposite.create(getWidgetFactory(), actionPage);
	}
	
	private void createEventPage() {
		Composite eventPage = detailsArea.createPage("org.jbpm.gd.jpdl.core.event");
		eventPage.setLayout(new FormLayout());
		eventConfigurationComposite = EventConfigurationComposite.create(getWidgetFactory(), eventPage);
	}
	
	private void createScriptPage() {
		Composite scriptPage = detailsArea.createPage("org.jbpm.gd.jpdl.core.script");
		scriptPage.setLayout(new FormLayout());
		scriptConfigurationComposite = ScriptConfigurationComposite.create(getWidgetFactory(), scriptPage);
	}
		
	private void createCreateTimerPage() {
		Composite createTimerPage = detailsArea.createPage("org.jbpm.gd.jpdl.core.createTimer");
		createTimerPage.setLayout(new FormLayout());
		createTimerConfigurationComposite = CreateTimerConfigurationComposite.create(getWidgetFactory(), createTimerPage);
	}
	
	private void createCancelTimerPage() {
		Composite cancelTimerPage = detailsArea.createPage("org.jbpm.gd.jpdl.core.cancelTimer");
		cancelTimerPage.setLayout(new FormLayout());
		cancelTimerConfigurationComposite = CancelTimerConfigurationComposite.create(getWidgetFactory(), cancelTimerPage);
	}
	
	private void createMailPage() {
		Composite mailPage = detailsArea.createPage("org.jbpm.gd.jpdl.core.mail");
		mailPage.setLayout(new FormLayout());
		mailConfigurationComposite = MailConfigurationComposite.create(getWidgetFactory(), mailPage);
	}
	
	private FormData createDetailsAreaLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(eventTree, 0);
		data.top = new FormAttachment(0, 0);
		data.right = new FormAttachment(100, 0);
		data.bottom = new FormAttachment(100, 0);
		return data;
	}
	
 	public void setInput(IWorkbenchPart part, ISelection selection) {
        super.setInput(part, selection);
        if (!(selection instanceof IStructuredSelection)) return;
        Object input = ((IStructuredSelection)selection).getFirstElement();
        if (input instanceof NotationElementGraphicalEditPart) {
        	AbstractNotationElement notationElement = ((NotationElementGraphicalEditPart)input).getNotationElement();
        	setEventContainer((EventContainer)notationElement.getSemanticElement());
        } else if (input instanceof JpdlOutlineEditPart) {
        	setEventContainer((EventContainer)((JpdlOutlineEditPart)input).getModel());
        }
    }
 	
 	private void clearControls() {
 		eventTree.removeAll();
 		detailsArea.showEmptyPage();
 	}
 	
 	private void setEventContainer(EventContainer newEventContainer) {
 		if (eventContainer == newEventContainer) return;
 		if (eventContainer != null) {
 			eventContainer.removePropertyChangeListener(this);
 		}
 		clearControls();
 		eventContainer = newEventContainer;
 		if (eventContainer != null) {
 			updateEventTree();
 			eventContainer.addPropertyChangeListener(this);
 		} 		
 	}
 	
 	private void updateEventTree() {
 		Event[] events = eventContainer.getEvents();
 		for (int i = 0; i < events.length; i++) {
 			TreeItem eventItem = new TreeItem(eventTree, SWT.NULL);
 			eventItem.setText(getLabel(events[i]));
 			eventItem.setData(events[i]);
 			eventItem.setImage(JpdlLabelProvider.getImage(events[i]));
 			eventItem.setExpanded(true);
 			ActionElement[] actionElements = events[i].getActionElements();
 			for (int j = 0; j < actionElements.length; j++) {
 				TreeItem actionElementItem = new TreeItem(eventItem, SWT.NULL);
 				actionElementItem.setText(getLabel(actionElements[j]));
 				actionElementItem.setData(actionElements[j]);
 				actionElementItem.setImage(JpdlLabelProvider.getImage(actionElements[j]));
 			}
 		}
 	}
 	
 	private String getLabel(ActionElement actionElement) {
 		String name = actionElement.getName();
 		if (name != null && !"".equals(name)) return name;
 		if (actionElement instanceof Action) {
 			String refName = ((Action)actionElement).getRefName();
 			if (refName != null && !"".equals(refName)) {
 				return refName;
 			}
 		}
 		return JpdlLabelProvider.getLabel(actionElement).toLowerCase();
 	}
 	
	public TabbedPropertySheetPage getTabbedPropertySheetPage() {
		return tabbedPropertySheetPage;
	}

	public void propertyChange(PropertyChangeEvent evt) {
		if (eventTree.isDisposed()) return;
		if ("eventAdd".equals(evt.getPropertyName())) {
			TreeItem treeItem = new TreeItem(eventTree, SWT.NULL);
			Event event = (Event)evt.getNewValue();
			treeItem.setText(getLabel(event));
			treeItem.setData(event);
			treeItem.setImage(JpdlLabelProvider.getImage(event));
			eventTree.setSelection(treeItem);
			eventTree.notifyListeners(SWT.Selection, new org.eclipse.swt.widgets.Event());
		} else if ("actionElementAdd".equals(evt.getPropertyName())) {
			TreeItem selection = eventTree.getSelection()[0];
			if (selection.getData() instanceof ActionElement) {
				selection = selection.getParentItem();
			}
			TreeItem treeItem = new TreeItem(selection, SWT.NULL);
			ActionElement actionElement = (ActionElement)evt.getNewValue();
			treeItem.setText(getLabel(actionElement));
			treeItem.setData(actionElement);
			treeItem.setImage(JpdlLabelProvider.getImage(actionElement));
			eventTree.setSelection(treeItem);
			eventTree.notifyListeners(SWT.Selection, new org.eclipse.swt.widgets.Event());
		} else if ("eventRemove".equals(evt.getPropertyName()) || 
				"actionElementRemove".equals(evt.getPropertyName())) {
			eventTree.getSelection()[0].dispose();
			eventTree.notifyListeners(SWT.Selection, new org.eclipse.swt.widgets.Event());
		} else if ("type".equals(evt.getPropertyName())) {
			String newValue = (String)evt.getNewValue();
			if (newValue == null || "".equals(newValue)) {
				newValue = "event";
			}
			eventTree.getSelection()[0].setText(newValue);
		} else if ("name".equals(evt.getPropertyName()) || "refName".equals(evt.getPropertyName())) {
			String newValue = (String)evt.getNewValue();
			if (newValue == null || "".equals(newValue)) {
				newValue = "action";
			}
			eventTree.getSelection()[0].setText(newValue);
		}
	}
	
	private String getLabel(Event event) {
		String label = event.getType();
		return label == null || "".equals(label) ? "event" : label;
	}
	
	public boolean shouldUseExtraSpace() {
		return true;
	}
	
	public EventContainer getEventContainer() {
		return eventContainer;
	}
	
	public Event getSelectedEvent() {
		return selectedEvent;
	}
	
	public ActionElement getSelectedActionElement() {
		return selectedActionElement;
	}

	public void aboutToBeShown() {
		actionBarContributor.activateContributions();
	}
	
	public void aboutToBeHidden() {
		actionBarContributor.deactivateContributions();
	}
	
//	public CommandStack getCommandStack() {
//		return (CommandStack)getPart().getSite().getWorkbenchWindow().getActivePage().getActiveEditor().getAdapter(CommandStack.class);
//		return (CommandStack)getPart().getAdapter(CommandStack.class);
//	}

}