"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const execa = require("execa");
var VersionHelper;
(function (VersionHelper) {
    VersionHelper.MINIMAL_OPENSHIFT_VERSION = '3.11';
    VersionHelper.MINIMAL_K8S_VERSION = '1.9';
    VersionHelper.MINIMAL_HELM_VERSION = '2.15';
    function getOpenShiftCheckVersionTask(flags) {
        return {
            title: 'Check OpenShift version',
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const actualVersion = yield getOpenShiftVersion();
                if (actualVersion) {
                    task.title = `${task.title}: Found ${actualVersion}.`;
                }
                else {
                    task.title = `${task.title}: Unknown.`;
                }
                if (!flags['skip-version-check'] && actualVersion) {
                    const checkPassed = checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_OPENSHIFT_VERSION);
                    if (!checkPassed) {
                        throw getError('OpenShift', actualVersion, VersionHelper.MINIMAL_OPENSHIFT_VERSION);
                    }
                }
            })
        };
    }
    VersionHelper.getOpenShiftCheckVersionTask = getOpenShiftCheckVersionTask;
    function getK8sCheckVersionTask(flags) {
        return {
            title: 'Check Kubernetes version',
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let actualVersion;
                switch (flags.platform) {
                    case 'minishift':
                    case 'openshift':
                    case 'crc':
                        actualVersion = yield getK8sVersionWithOC();
                        break;
                    default:
                        actualVersion = yield getK8sVersionWithKubectl();
                }
                if (actualVersion) {
                    task.title = `${task.title}: Found ${actualVersion}.`;
                }
                else {
                    task.title = `${task.title}: Unknown.`;
                }
                if (!flags['skip-version-check'] && actualVersion) {
                    const checkPassed = checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_K8S_VERSION);
                    if (!checkPassed) {
                        throw getError('Kubernetes', actualVersion, VersionHelper.MINIMAL_K8S_VERSION);
                    }
                }
            })
        };
    }
    VersionHelper.getK8sCheckVersionTask = getK8sCheckVersionTask;
    function getOpenShiftVersion() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return getVersionWithOC('openshift ');
        });
    }
    VersionHelper.getOpenShiftVersion = getOpenShiftVersion;
    function getK8sVersionWithOC() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return getVersionWithOC('kubernetes ');
        });
    }
    VersionHelper.getK8sVersionWithOC = getK8sVersionWithOC;
    function getK8sVersionWithKubectl() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return getVersionWithKubectl('Server Version: ');
        });
    }
    VersionHelper.getK8sVersionWithKubectl = getK8sVersionWithKubectl;
    function checkMinimalK8sVersion(actualVersion) {
        return checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_K8S_VERSION);
    }
    VersionHelper.checkMinimalK8sVersion = checkMinimalK8sVersion;
    function checkMinimalOpenShiftVersion(actualVersion) {
        return checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_OPENSHIFT_VERSION);
    }
    VersionHelper.checkMinimalOpenShiftVersion = checkMinimalOpenShiftVersion;
    function checkMinimalHelmVersion(actualVersion) {
        return checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_HELM_VERSION);
    }
    VersionHelper.checkMinimalHelmVersion = checkMinimalHelmVersion;
    /**
     * Compare versions and return true if actual version is greater or equal to minimal.
     * The comparison will be done by major and minor versions.
     */
    function checkMinimalVersion(actual, minimal) {
        actual = removeVPrefix(actual);
        let vers = actual.split('.');
        const actualMajor = parseInt(vers[0], 10);
        const actualMinor = parseInt(vers[1], 10);
        minimal = removeVPrefix(minimal);
        vers = minimal.split('.');
        const minimalMajor = parseInt(vers[0], 10);
        const minimalMinor = parseInt(vers[1], 10);
        return (actualMajor > minimalMajor || (actualMajor === minimalMajor && actualMinor >= minimalMinor));
    }
    VersionHelper.checkMinimalVersion = checkMinimalVersion;
    function getError(actualVersion, minimalVersion, component) {
        return new Error(`The minimal supported version of ${component} is '${minimalVersion} but found '${actualVersion}'. To bypass version check use '--skip-version-check' flag.`);
    }
    VersionHelper.getError = getError;
    function getVersionWithOC(versionPrefix) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const command = 'oc';
            const args = ['version'];
            const { stdout } = yield execa(command, args, { timeout: 60000 });
            return stdout.split('\n').filter(value => value.startsWith(versionPrefix)).map(value => value.substring(versionPrefix.length))[0];
        });
    }
    function getVersionWithKubectl(versionPrefix) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const command = 'kubectl';
            const args = ['version', '--short'];
            const { stdout } = yield execa(command, args, { timeout: 60000 });
            return stdout.split('\n').filter(value => value.startsWith(versionPrefix)).map(value => value.substring(versionPrefix.length))[0];
        });
    }
    function removeVPrefix(version) {
        return version.startsWith('v') ? version.substring(1) : version;
    }
})(VersionHelper = exports.VersionHelper || (exports.VersionHelper = {}));
//# sourceMappingURL=version.js.map