"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const fs = require("fs-extra");
const Listr = require("listr");
const notifier = require("node-notifier");
const os = require("os");
const path = require("path");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const che_1 = require("../../tasks/che");
const installer_1 = require("../../tasks/installers/installer");
const api_1 = require("../../tasks/platforms/api");
const platform_1 = require("../../tasks/platforms/platform");
class Start extends command_1.Command {
    static getTemplatesDir() {
        // return local templates folder if present
        const TEMPLATES = 'templates';
        const templatesDir = path.resolve(TEMPLATES);
        const exists = fs.pathExistsSync(templatesDir);
        if (exists) {
            return TEMPLATES;
        }
        // else use the location from modules
        return path.join(__dirname, '../../../templates');
    }
    static setPlaformDefaults(flags) {
        flags.installer = 'operator';
    }
    checkPlatformCompatibility(flags) {
        // matrix checks
        if (flags.installer === 'operator' && flags['che-operator-cr-yaml']) {
            const ignoredFlags = [];
            flags['plugin-registry-url'] && ignoredFlags.push('--plugin-registry-urlomain');
            flags['devfile-registry-url'] && ignoredFlags.push('--devfile-registry-url');
            flags['postgres-pvc-storage-class-name'] && ignoredFlags.push('--postgres-pvc-storage-class-name');
            flags['workspace-pvc-storage-class-name'] && ignoredFlags.push('--workspace-pvc-storage-class-name');
            flags['self-signed-cert'] && ignoredFlags.push('--self-signed-cert');
            flags['os-oauth'] && ignoredFlags.push('--os-oauth');
            flags.tls && ignoredFlags.push('--tls');
            flags.cheimage && ignoredFlags.push('--cheimage');
            flags.debug && ignoredFlags.push('--debug');
            flags.domain && ignoredFlags.push('--domain');
            if (ignoredFlags.length) {
                this.warn(`--che-operator-cr-yaml is used. The following flag(s) will be ignored: ${ignoredFlags.join('\t')}`);
            }
        }
        if (flags.installer) {
            if (flags.installer === 'minishift-addon') {
                if (flags.platform !== 'minishift') {
                    this.error(`🛑 Current platform is ${flags.platform}. Minishift-addon is only available for Minishift.`);
                }
            }
            else if (flags.installer === 'helm') {
                if (flags.platform !== 'k8s' && flags.platform !== 'minikube' && flags.platform !== 'microk8s' && flags.platform !== 'docker-desktop') {
                    this.error(`🛑 Current platform is ${flags.platform}. Helm installer is only available on top of Kubernetes flavor platform (including Minikube, Docker Desktop).`);
                }
            }
            if (flags['os-oauth']) {
                if (flags.platform !== 'openshift' && flags.platform !== 'minishift' && flags.platform !== 'crc') {
                    this.error(`You requested to enable OpenShift OAuth but the platform doesn\'t seem to be OpenShift. Platform is ${flags.platform}.`);
                }
                if (flags.installer !== 'operator') {
                    this.error(`You requested to enable OpenShift OAuth but that's only possible when using the operator as installer. The current installer is ${flags.installer}. To use the operator add parameter "--installer operator".`);
                }
            }
        }
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Start);
            const ctx = {};
            ctx.directory = path.resolve(flags.directory ? flags.directory : path.resolve(os.tmpdir(), 'crwctl-logs', Date.now().toString()));
            const listrOptions = { renderer: flags['listr-renderer'], collapse: false, showSubtasks: true };
            const cheTasks = new che_1.CheTasks(flags);
            const platformTasks = new platform_1.PlatformTasks();
            const installerTasks = new installer_1.InstallerTasks();
            const apiTasks = new api_1.ApiTasks();
            // Platform Checks
            let platformCheckTasks = new Listr(platformTasks.preflightCheckTasks(flags, this), listrOptions);
            // Checks if CodeReady Workspaces is already deployed
            let preInstallTasks = new Listr(undefined, listrOptions);
            preInstallTasks.add(apiTasks.testApiTasks(flags, this));
            preInstallTasks.add({
                title: '👀  Looking for an already existing CodeReady Workspaces instance',
                task: () => new Listr(cheTasks.checkIfCheIsInstalledTasks(flags, this))
            });
            Start.setPlaformDefaults(flags);
            let installTasks = new Listr(installerTasks.installTasks(flags, this), listrOptions);
            const startDeployedCheTasks = new Listr([{
                    title: '👀  Starting already deployed CodeReady Workspaces',
                    task: () => new Listr(cheTasks.scaleCheUpTasks(this))
                }], listrOptions);
            // Post Install Checks
            const postInstallTasks = new Listr([{
                    title: '✅  Post installation checklist',
                    task: () => new Listr(cheTasks.waitDeployedChe(flags, this))
                }], listrOptions);
            const logsTasks = new Listr([{
                    title: 'Start following logs',
                    task: () => new Listr(cheTasks.serverLogsTasks(flags, true))
                }], listrOptions);
            const eventTasks = new Listr([{
                    title: 'Start following events',
                    task: () => new Listr(cheTasks.namespaceEventsTask(flags.chenamespace, this, true))
                }], listrOptions);
            try {
                yield preInstallTasks.run(ctx);
                if (!ctx.isCheDeployed) {
                    this.checkPlatformCompatibility(flags);
                    yield platformCheckTasks.run(ctx);
                    this.log(`CodeReady Workspaces logs will be available in '${ctx.directory}'`);
                    yield logsTasks.run(ctx);
                    yield eventTasks.run(ctx);
                    yield installTasks.run(ctx);
                }
                else if (!ctx.isCheReady
                    || (ctx.isPostgresDeployed && !ctx.isPostgresReady)
                    || (ctx.isKeycloakDeployed && !ctx.isKeycloakReady)
                    || (ctx.isPluginRegistryDeployed && !ctx.isPluginRegistryReady)
                    || (ctx.isDevfileRegistryDeployed && !ctx.isDevfileRegistryReady)) {
                    if (flags.platform || flags.installer) {
                        this.warn('Deployed CodeReady Workspaces is found and the specified installation parameters will be ignored');
                    }
                    // perform CodeReady Workspaces start task if there is any component that is not ready
                    yield startDeployedCheTasks.run(ctx);
                }
                yield postInstallTasks.run(ctx);
                this.log('Command server:start has completed successfully.');
            }
            catch (err) {
                this.error(`${err}\nInstallation failed, check logs in '${ctx.directory}'`);
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command server:start has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Start;
Start.description = 'start CodeReady Workspaces server';
Start.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    'listr-renderer': common_flags_1.listrRenderer,
    'deployment-name': common_flags_1.cheDeployment,
    cheimage: flags_1.string({
        char: 'i',
        description: 'CodeReady Workspaces server container image',
        default: constants_1.DEFAULT_CHE_IMAGE,
        env: 'CHE_CONTAINER_IMAGE'
    }),
    templates: flags_1.string({
        char: 't',
        description: 'Path to the templates folder',
        default: Start.getTemplatesDir(),
        env: 'CHE_TEMPLATES_FOLDER'
    }),
    'devfile-registry-url': flags_1.string({
        description: 'The URL of the external Devfile registry.',
        env: 'CHE_WORKSPACE_DEVFILE__REGISTRY__URL'
    }),
    'plugin-registry-url': flags_1.string({
        description: 'The URL of the external plugin registry.',
        env: 'CHE_WORKSPACE_PLUGIN__REGISTRY__URL'
    }),
    cheboottimeout: flags_1.string({
        char: 'o',
        description: 'CodeReady Workspaces server bootstrap timeout (in milliseconds)',
        default: '40000',
        required: true,
        env: 'CHE_SERVER_BOOT_TIMEOUT'
    }),
    k8spodwaittimeout: flags_1.string({
        description: 'Waiting time for Pod Wait Timeout Kubernetes (in milliseconds)',
        default: '300000'
    }),
    k8spodreadytimeout: flags_1.string({
        description: 'Waiting time for Pod Ready Kubernetes (in milliseconds)',
        default: '130000'
    }),
    multiuser: command_1.flags.boolean({
        char: 'm',
        description: 'Starts CodeReady Workspaces in multi-user mode',
        default: false
    }),
    tls: command_1.flags.boolean({
        char: 's',
        description: `Enable TLS encryption.
                    Note that for kubernetes 'che-tls' with TLS certificate must be created in the configured namespace.
                    For OpenShift, router will use default cluster certificates.`
    }),
    'self-signed-cert': command_1.flags.boolean({
        description: `Authorize usage of self signed certificates for encryption.
                    This is the flag for CodeReady Workspaces to propagate the certificate to components, so they will trust it.
                    Note that \`che-tls\` secret with CA certificate must be created in the configured namespace.`,
        default: false
    }),
    platform: flags_1.string({
        char: 'p',
        description: 'Type of OpenShift platform. Valid values are \"openshift\", \"crc (for CodeReady Containers)\".',
        options: ['openshift', 'crc'],
        default: 'openshift'
    }),
    installer: flags_1.string({
        char: 'a',
        description: 'Installer type',
        options: ['operator'],
        default: 'operator'
    }),
    domain: flags_1.string({
        char: 'b',
        description: 'Domain of the Kubernetes cluster (e.g. example.k8s-cluster.com or <local-ip>.nip.io)',
        default: ''
    }),
    debug: flags_1.boolean({
        description: 'Enables the debug mode for CodeReady Workspaces server. To debug CodeReady Workspaces server from localhost use \'server:debug\' command.',
        default: false
    }),
    'os-oauth': command_1.flags.boolean({
        description: 'Enable use of OpenShift credentials to log into CodeReady Workspaces',
        default: false
    }),
    'che-operator-image': flags_1.string({
        description: 'Container image of the operator. This parameter is used only when the installer is the operator',
        default: constants_1.DEFAULT_CHE_OPERATOR_IMAGE
    }),
    'che-operator-cr-yaml': flags_1.string({
        description: 'Path to a yaml file that defines a CheCluster used by the operator. This parameter is used only when the installer is the operator.',
        default: ''
    }),
    'che-operator-cr-patch-yaml': flags_1.string({
        description: 'Path to a yaml file that overrides the default values in CheCluster CR used by the operator. This parameter is used only when the installer is the operator.',
        default: ''
    }),
    directory: flags_1.string({
        char: 'd',
        description: 'Directory to store logs into',
        env: 'CHE_LOGS'
    }),
    'workspace-pvc-storage-class-name': flags_1.string({
        description: 'persistent volume(s) storage class name to use to store CodeReady Workspaces workspaces data',
        env: 'CHE_INFRA_KUBERNETES_PVC_STORAGE__CLASS__NAME',
        default: ''
    }),
    'postgres-pvc-storage-class-name': flags_1.string({
        description: 'persistent volume storage class name to use to store CodeReady Workspaces postgres database',
        default: ''
    }),
    'skip-version-check': command_1.flags.boolean({
        description: 'Skip minimal versions check.',
        default: false
    })
};
//# sourceMappingURL=start.js.map