"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const cli_ux_1 = require("cli-ux");
const execa = require("execa");
const fs_1 = require("fs");
const fs_extra_1 = require("fs-extra");
const yaml = require("js-yaml");
const Listr = require("listr");
const path = require("path");
const che_1 = require("../../api/che");
const kube_1 = require("../../api/kube");
class OperatorTasks {
    constructor() {
        this.operatorServiceAccount = 'codeready-operator';
        this.operatorRole = 'codeready-operator';
        this.operatorClusterRole = 'codeready-operator';
        this.operatorRoleBinding = 'codeready-operator';
        this.operatorClusterRoleBinding = 'codeready-operator';
        this.cheClusterCrd = 'checlusters.org.eclipse.che';
        this.operatorName = 'codeready-operator';
        this.operatorCheCluster = 'codeready-workspaces';
        this.resourcesPath = '';
    }
    /**
     * Returns tasks list which perform preflight platform checks.
     */
    startTasks(flags, command) {
        const che = new che_1.CheHelper(flags);
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Copying operator resources',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    this.resourcesPath = yield this.copyCheOperatorResources(flags.templates, command.config.cacheDir);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Create Namespace (${flags.chenamespace})`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield che.cheNamespaceExist(flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else if (flags.platform === 'minikube' || flags.platform === 'k8s' || flags.platform === 'microk8s') {
                        yield execa(`kubectl create namespace ${flags.chenamespace}`, { shell: true });
                        task.title = `${task.title}...done.`;
                    }
                    else if (flags.platform === 'minishift' || flags.platform === 'openshift' || flags.platform === 'crc') {
                        yield execa(`oc new-project ${flags.chenamespace}`, { shell: true });
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = this.resourcesPath + 'service_account.yaml';
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create Role ${this.operatorRole} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleExist(this.operatorRole, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = this.resourcesPath + 'role.yaml';
                        const statusCode = yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRole ${this.operatorClusterRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleExist(this.operatorClusterRole);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = this.resourcesPath + 'cluster_role.yaml';
                        const statusCode = yield kube.createClusterRoleFromFile(yamlFilePath);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create RoleBinding ${this.operatorRoleBinding} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.operatorRoleBinding, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = this.resourcesPath + 'role_binding.yaml';
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create ClusterRoleBinding ${this.operatorClusterRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleBindingExist(this.operatorRoleBinding);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(this.operatorClusterRoleBinding, this.operatorServiceAccount, flags.chenamespace, this.operatorClusterRole);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create CRD ${this.cheClusterCrd}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.crdExist(this.cheClusterCrd);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        const yamlFilePath = this.resourcesPath + 'crds/org_v1_che_crd.yaml';
                        yield kube.createCrdFromFile(yamlFilePath);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Create deployment ${this.operatorName} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(this.operatorName, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        yield kube.createDeploymentFromFile(this.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...done.`;
                    }
                })
            },
            {
                title: `Create CodeReady Workspaces cluster ${this.operatorCheCluster} in namespace ${flags.chenamespace}`,
                task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.cheClusterExist(this.operatorCheCluster, flags.chenamespace);
                    if (exist) {
                        task.title = `${task.title}...It already exists.`;
                    }
                    else {
                        // CodeReady Workspaces operator supports only Multi-User Che
                        ctx.isCheDeployed = true;
                        ctx.isPostgresDeployed = true;
                        ctx.isKeycloakDeployed = true;
                        // plugin and devfile registry will be deployed only when external ones are not configured
                        ctx.isPluginRegistryDeployed = !flags['plugin-registry-url'];
                        ctx.isDevfileRegistryDeployed = !flags['devfile-registry-url'];
                        const yamlFilePath = flags['che-operator-cr-yaml'] === '' ? this.resourcesPath + 'crds/org_v1_che_cr.yaml' : flags['che-operator-cr-yaml'];
                        const cr = yield kube.createCheClusterFromFile(yamlFilePath, flags, flags['che-operator-cr-yaml'] === '');
                        ctx.isKeycloakReady = ctx.isKeycloakReady || cr.spec.auth.externalIdentityProvider;
                        ctx.isPostgresReady = ctx.isPostgresReady || cr.spec.database.externalDb;
                        ctx.isDevfileRegistryReady = ctx.isDevfileRegistryReady || cr.spec.server.externalDevfileRegistry;
                        ctx.isPluginRegistryReady = ctx.isPluginRegistryReady || cr.spec.server.externalPluginRegistry;
                        task.title = `${task.title}...done.`;
                    }
                })
            }
        ], { renderer: flags['listr-renderer'] });
    }
    preUpdateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Checking versions compatibility before updating',
                task: (ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const operatorDeployment = yield kube.getDeployment(this.operatorName, flags.chenamespace);
                    if (!operatorDeployment) {
                        command.error(`${this.operatorName} deployment is not found in namespace ${flags.chenamespace}.\nProbably CodeReady Workspaces was initially deployed with another installer`);
                        return;
                    }
                    const deployedCheOperator = this.retrieveContainerImage(operatorDeployment);
                    const deployedCheOperatorImageAndTag = deployedCheOperator.split(':', 2);
                    ctx.deployedCheOperatorImage = deployedCheOperatorImageAndTag[0];
                    ctx.deployedCheOperatorTag = deployedCheOperatorImageAndTag.length === 2 ? deployedCheOperatorImageAndTag[1] : 'latest';
                    const newCheOperatorImageAndTag = flags['che-operator-image'].split(':', 2);
                    ctx.newCheOperatorImage = newCheOperatorImageAndTag[0];
                    ctx.newCheOperatorTag = newCheOperatorImageAndTag.length === 2 ? newCheOperatorImageAndTag[1] : 'latest';
                })
            }
        ]);
    }
    updateTasks(flags, command) {
        const kube = new kube_1.KubeHelper(flags);
        return new Listr([
            {
                title: 'Copying operator resources',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    this.resourcesPath = yield this.copyCheOperatorResources(flags.templates, command.config.cacheDir);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Updating ServiceAccount ${this.operatorServiceAccount} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace);
                    const yamlFilePath = this.resourcesPath + 'service_account.yaml';
                    if (exist) {
                        yield kube.replaceServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createServiceAccountFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating Role ${this.operatorRole} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleExist(this.operatorRole, flags.chenamespace);
                    const yamlFilePath = this.resourcesPath + 'role.yaml';
                    if (exist) {
                        const statusCode = yield kube.replaceRoleFromFile(yamlFilePath, flags.chenamespace);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        const statusCode = yield kube.createRoleFromFile(yamlFilePath, flags.chenamespace);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRole ${this.operatorClusterRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleExist(this.operatorClusterRole);
                    const yamlFilePath = this.resourcesPath + 'cluster_role.yaml';
                    if (exist) {
                        const statusCode = yield kube.replaceClusterRoleFromFile(yamlFilePath);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        const statusCode = yield kube.createClusterRoleFromFile(yamlFilePath);
                        if (statusCode === 403) {
                            command.error('ERROR: It looks like you don\'t have enough privileges. You need to grant more privileges to current user or use a different user. If you are using minishift you can "oc login -u system:admin"');
                        }
                        task.title = `${task.title}...created a new one.`;
                    }
                })
            },
            {
                title: `Updating RoleBinding ${this.operatorRoleBinding} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.roleBindingExist(this.operatorRoleBinding, flags.chenamespace);
                    const yamlFilePath = this.resourcesPath + 'role_binding.yaml';
                    if (exist) {
                        yield kube.replaceRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createRoleBindingFromFile(yamlFilePath, flags.chenamespace);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating ClusterRoleBinding ${this.operatorClusterRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.clusterRoleBindingExist(this.operatorRoleBinding);
                    if (exist) {
                        yield kube.replaceClusterRoleBinding(this.operatorClusterRoleBinding, this.operatorServiceAccount, flags.chenamespace, this.operatorClusterRole);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createClusterRoleBinding(this.operatorClusterRoleBinding, this.operatorServiceAccount, flags.chenamespace, this.operatorClusterRole);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: `Updating CodeReady Workspaces cluster CRD ${this.cheClusterCrd}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const crd = yield kube.getCrd(this.cheClusterCrd);
                    const yamlFilePath = this.resourcesPath + 'crds/org_v1_che_crd.yaml';
                    if (crd) {
                        if (!crd.metadata || !crd.metadata.resourceVersion) {
                            throw new Error(`Fetched CRD ${this.cheClusterCrd} without resource version`);
                        }
                        yield kube.replaceCrdFromFile(yamlFilePath, crd.metadata.resourceVersion);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createCrdFromFile(yamlFilePath);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting 5 seconds for the new Kubernetes resources to get flushed',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(5000);
                    task.title = `${task.title}...done.`;
                })
            },
            {
                title: `Updating deployment ${this.operatorName} in namespace ${flags.chenamespace}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    const exist = yield kube.deploymentExist(this.operatorName, flags.chenamespace);
                    if (exist) {
                        yield kube.replaceDeploymentFromFile(this.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...updated.`;
                    }
                    else {
                        yield kube.createDeploymentFromFile(this.resourcesPath + 'operator.yaml', flags.chenamespace, flags['che-operator-image']);
                        task.title = `${task.title}...created new one.`;
                    }
                })
            },
            {
                title: 'Waiting newer operator to be run',
                task: (_ctx, _task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    yield cli_ux_1.cli.wait(1000);
                    yield kube.waitLatestReplica(this.operatorName, flags.chenamespace);
                })
            }
        ], { renderer: flags['listr-renderer'] });
    }
    /**
     * Returns list of tasks which remove CodeReady Workspaces operator related resources
     */
    deleteTasks(flags) {
        let kh = new kube_1.KubeHelper(flags);
        return [{
                title: `Delete the CR ${this.operatorCheCluster} of type ${this.cheClusterCrd}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if ((yield kh.crdExist(this.cheClusterCrd)) &&
                        (yield kh.cheClusterExist(this.operatorCheCluster, flags.chenamespace))) {
                        yield kh.deleteCheCluster(this.operatorCheCluster, flags.chenamespace);
                        yield cli_ux_1.cli.wait(2000); //wait a couple of secs for the finalizers to be executed
                        task.title = yield `${task.title}...OK`;
                    }
                    else {
                        task.title = yield `${task.title}...CR not found`;
                    }
                })
            },
            {
                title: `Delete CRD ${this.cheClusterCrd}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.crdExist(this.cheClusterCrd)) {
                        yield kh.deleteCrd(this.cheClusterCrd);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete role binding ${this.operatorRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleBindingExist(this.operatorRoleBinding, flags.chenamespace)) {
                        yield kh.deleteRoleBinding(this.operatorRoleBinding, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete role ${this.operatorRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleExist(this.operatorRole, flags.chenamespace)) {
                        yield kh.deleteRole(this.operatorRole, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete cluster role binding ${this.operatorClusterRoleBinding}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.clusterRoleBindingExist(this.operatorClusterRoleBinding)) {
                        yield kh.deleteClusterRoleBinding(this.operatorClusterRoleBinding);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete cluster role ${this.operatorClusterRole}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.clusterRoleExist(this.operatorClusterRole)) {
                        yield kh.deleteClusterRole(this.operatorClusterRole);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete server and workspace rolebindings',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.roleBindingExist('che', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che', flags.chenamespace);
                    }
                    if (yield kh.roleBindingExist('che-workspace-exec', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che-workspace-exec', flags.chenamespace);
                    }
                    if (yield kh.roleBindingExist('che-workspace-view', flags.chenamespace)) {
                        yield kh.deleteRoleBinding('che-workspace-view', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: `Delete service accounts ${this.operatorServiceAccount}`,
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.serviceAccountExist(this.operatorServiceAccount, flags.chenamespace)) {
                        yield kh.deleteServiceAccount(this.operatorServiceAccount, flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
            {
                title: 'Delete PVC codeready-operator',
                task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                    if (yield kh.persistentVolumeClaimExist('codeready-operator', flags.chenamespace)) {
                        yield kh.deletePersistentVolumeClaim('codeready-operator', flags.chenamespace);
                    }
                    task.title = yield `${task.title}...OK`;
                })
            },
        ];
    }
    copyCheOperatorResources(templatesDir, cacheDir) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const srcDir = path.join(templatesDir, '/codeready-workspaces-operator/');
            const destDir = path.join(cacheDir, '/templates/codeready-workspaces-operator/');
            yield fs_extra_1.remove(destDir);
            yield fs_extra_1.mkdirp(destDir);
            yield fs_extra_1.copy(srcDir, destDir);
            return destDir;
        });
    }
    evaluateTemplateOperatorImage(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (flags['che-operator-image']) {
                return flags['che-operator-image'];
            }
            else {
                const filePath = flags.templates + '/codeready-workspaces-operator/operator.yaml';
                const yamlFile = fs_1.readFileSync(filePath);
                const yamlDeployment = yaml.safeLoad(yamlFile.toString());
                return yamlDeployment.spec.template.spec.containers[0].image;
            }
        });
    }
    retrieveContainerImage(deployment) {
        const containers = deployment.spec.template.spec.containers;
        const namespace = deployment.metadata.namespace;
        const name = deployment.metadata.name;
        if (containers.length === 0) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. Containers list are empty`);
        }
        if (containers.length > 1) {
            throw new Error(`Can not evaluate image of ${namespace}/${name} deployment. It has multiple containers`);
        }
        const container = containers[0];
        if (!container.image) {
            throw new Error(`Container ${container.name} in deployment ${namespace}/${name} must have image specified`);
        }
        return container.image;
    }
}
exports.OperatorTasks = OperatorTasks;
//# sourceMappingURL=operator.js.map