"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const che_1 = require("../../api/che");
const common_flags_1 = require("../../common-flags");
class Start extends command_1.Command {
    checkToken(flags, ctx) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (ctx.isAuthEnabled && !flags['access-token']) {
                this.error('E_AUTH_REQUIRED - CodeReady Workspaces authentication is enabled and an access token need to be provided (flag --access-token).');
            }
        });
    }
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Start);
            const Listr = require('listr');
            const notifier = require('node-notifier');
            const che = new che_1.CheHelper(flags);
            if (!flags.devfile && !flags.workspaceconfig) {
                this.error('workspace:start command is expecting a devfile or workspace configuration parameter.');
            }
            const tasks = new Listr([
                {
                    title: 'Retrieving CodeReady Workspaces server URL',
                    task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        ctx.cheURL = yield che.cheURL(flags.chenamespace);
                        task.title = yield `${task.title}...${ctx.cheURL}`;
                    })
                },
                {
                    title: 'Verify if CodeReady Workspaces server is running',
                    task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        if (!(yield che.isCheServerReady(ctx.cheURL))) {
                            this.error(`E_SRV_NOT_RUNNING - CodeReady Workspaces server is not available by ${ctx.cheURL}`, { code: 'E_SRV_NOT_RUNNNG' });
                        }
                        const status = yield che.getCheServerStatus(ctx.cheURL);
                        ctx.isAuthEnabled = yield che.isAuthenticationEnabled(ctx.cheURL);
                        const auth = ctx.isAuthEnabled ? '(auth enabled)' : '(auth disabled)';
                        task.title = yield `${task.title}...${status} ${auth}`;
                    })
                },
                {
                    title: `Create workspace from Devfile ${flags.devfile}`,
                    enabled: () => flags.devfile !== undefined,
                    task: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield this.checkToken(flags, ctx);
                        ctx.workspaceIdeURL = yield che.createWorkspaceFromDevfile(flags.chenamespace, flags.devfile, flags.name, flags['access-token']);
                    })
                },
                {
                    title: `Create workspace from Workspace Config ${flags.workspaceconfig}`,
                    enabled: () => flags.workspaceconfig !== undefined,
                    task: (ctx) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                        yield this.checkToken(flags, ctx);
                        ctx.workspaceIdeURL = yield che.createWorkspaceFromWorkspaceConfig(flags.chenamespace, flags.workspaceconfig, flags['access-token']);
                    })
                },
            ], { renderer: flags['listr-renderer'] });
            try {
                let ctx = yield tasks.run();
                this.log('\nWorkspace IDE URL:');
                cli_ux_1.cli.url(ctx.workspaceIdeURL, ctx.workspaceIdeURL);
            }
            catch (err) {
                this.error(err);
            }
            notifier.notify({
                title: 'crwctl',
                message: 'Command workspace:start has completed successfully.'
            });
            this.exit(0);
        });
    }
}
exports.default = Start;
Start.description = 'create and start a workspace';
Start.flags = {
    help: command_1.flags.help({ char: 'h' }),
    chenamespace: common_flags_1.cheNamespace,
    devfile: flags_1.string({
        char: 'f',
        description: 'path or URL to a valid devfile',
        env: 'DEVFILE_PATH',
        required: false,
    }),
    workspaceconfig: flags_1.string({
        char: 'w',
        description: 'path to a valid workspace configuration json file',
        env: 'WORKSPACE_CONFIG_JSON_PATH',
        required: false,
    }),
    name: flags_1.string({
        description: 'workspace name: overrides the workspace name to use instead of the one defined in the devfile. Works only for devfile',
        required: false,
    }),
    'access-token': common_flags_1.accessToken,
    'listr-renderer': common_flags_1.listrRenderer
};
//# sourceMappingURL=start.js.map