package org.infinispan.xsite.statetransfer.failures;

import org.infinispan.Cache;
import org.infinispan.commands.remote.CacheRpcCommand;
import org.infinispan.commons.CacheException;
import org.infinispan.configuration.cache.BackupConfigurationBuilder;
import org.infinispan.distribution.MagicKey;
import org.infinispan.manager.CacheContainer;
import org.infinispan.remoting.InboundInvocationHandler;
import org.infinispan.remoting.responses.ExceptionResponse;
import org.infinispan.remoting.transport.Address;
import org.infinispan.remoting.transport.Transport;
import org.infinispan.remoting.transport.jgroups.CommandAwareRpcDispatcher;
import org.infinispan.remoting.transport.jgroups.JGroupsTransport;
import org.infinispan.xsite.BackupReceiver;
import org.infinispan.xsite.BackupReceiverDelegator;
import org.infinispan.xsite.BackupReceiverRepository;
import org.infinispan.xsite.BackupReceiverRepositoryDelegator;
import org.infinispan.xsite.statetransfer.XSiteState;
import org.infinispan.xsite.statetransfer.XSiteStateConsumer;
import org.infinispan.xsite.statetransfer.XSiteStatePushCommand;
import org.infinispan.xsite.statetransfer.XSiteStateTransferManager;
import org.jgroups.blocks.Response;
import org.testng.AssertJUnit;
import org.testng.annotations.Test;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;

import static org.infinispan.test.TestingUtil.WrapFactory;
import static org.infinispan.test.TestingUtil.extractComponent;
import static org.infinispan.test.TestingUtil.extractGlobalComponent;
import static org.infinispan.test.TestingUtil.replaceComponent;
import static org.infinispan.test.TestingUtil.replaceField;
import static org.infinispan.test.TestingUtil.wrapComponent;
import static org.infinispan.test.TestingUtil.wrapGlobalComponent;

/**
 * Tests the multiple retry mechanism implemented in Cross-Site replication state transfer.
 *
 * @author Pedro Ruivo
 * @since 7.1
 */
@Test(groups = "functional", testName = "xsite.statetransfer.failures.RetryMechanismTest")
public class RetryMechanismTest extends AbstractTopologyChangeTest {

   private static final String VALUE = "value";

   /**
    * Simple scenario where the primary owner throws an exception. The NYC site master retries and at the 3rd retry, it
    * will apply the data (test retry in NYC).
    */
   public void testExceptionWithSuccessfulRetry() {
      takeSiteOffline(LON, NYC);
      final Object key = new MagicKey(cache(NYC, 1));
      final FailureHandler handler = FailureHandler.replaceOn(cache(NYC, 1));
      final CounterBackupReceiverRepository counterRepository = CounterBackupReceiverRepository.replaceOn(cache(NYC, 0).getCacheManager());

      cache(LON, 0).put(key, VALUE);

      handler.fail(3); //it fails 3 times and then succeeds.

      startStateTransfer(cache(LON, 0), NYC);
      assertOnline(LON, NYC);

      awaitXSiteStateSent(LON);
      awaitXSiteStateReceived(NYC);

      AssertJUnit.assertEquals(0, handler.remainingFails());
      AssertJUnit.assertEquals(1, counterRepository.counter.get());
      assertInSite(NYC, new AssertCondition<Object, Object>() {
         @Override
         public void assertInCache(Cache<Object, Object> cache) {
            AssertJUnit.assertEquals(VALUE, cache.get(key));
         }
      });
   }

   /**
    * Simple scenario where the primary owner always throws an exception. The state transfer will not be successful
    * (test retry in NYC and LON).
    */
   public void testExceptionWithFailedRetry() {
      takeSiteOffline(LON, NYC);
      final Object key = new MagicKey(cache(NYC, 1));
      final FailureHandler handler = FailureHandler.replaceOn(cache(NYC, 1));
      final CounterBackupReceiverRepository counterRepository = CounterBackupReceiverRepository.replaceOn(cache(NYC, 0).getCacheManager());

      cache(LON, 0).put(key, VALUE);

      handler.failAlways();

      startStateTransfer(cache(LON, 0), NYC);
      assertOnline(LON, NYC);

      awaitXSiteStateSent(LON);
      awaitXSiteStateReceived(NYC);

      assertXSiteStatus(LON, NYC, XSiteStateTransferManager.STATUS_ERROR);

      AssertJUnit.assertEquals(3 /*max_retry + 1*/, counterRepository.counter.get());
      assertInSite(NYC, new AssertCondition<Object, Object>() {
         @Override
         public void assertInCache(Cache<Object, Object> cache) {
            AssertJUnit.assertNull(cache.get(key));
         }
      });
   }

   /**
    * Simple scenario where the primary owner leaves the cluster and the site master will apply the data locally (tests
    * retry in NYC, from remote to local).
    */
   public void testRetryLocally() throws ExecutionException, InterruptedException {
      takeSiteOffline(LON, NYC);
      final Object key = new MagicKey(cache(NYC, 1));
      final DiscardHandler handler = DiscardHandler.replaceOn(cache(NYC, 1));
      final CounterBackupReceiverRepository counterRepository = CounterBackupReceiverRepository.replaceOn(cache(NYC, 0).getCacheManager());

      cache(LON, 0).put(key, VALUE);

      startStateTransfer(cache(LON, 0), NYC);
      assertOnline(LON, NYC);

      eventually(new Condition() {
         @Override
         public boolean isSatisfied() throws Exception {
            return handler.discarded;
         }
      });

      triggerTopologyChange(NYC, 1).get();

      awaitXSiteStateSent(LON);
      awaitXSiteStateReceived(NYC);

      AssertJUnit.assertEquals(1, counterRepository.counter.get());

      assertInSite(NYC, new AssertCondition<Object, Object>() {
         @Override
         public void assertInCache(Cache<Object, Object> cache) {
            AssertJUnit.assertEquals(VALUE, cache.get(key));
         }
      });
   }

   /**
    * Simple scenario where the primary owner leaves the cluster and the NYC site master will apply the data locally.
    * The 1st and the 2nd time will fail and only the 3rd will succeed (testing local retry)
    */
   public void testMultipleRetryLocally() throws ExecutionException, InterruptedException {
      takeSiteOffline(LON, NYC);
      final Object key = new MagicKey(cache(NYC, 1));
      final DiscardHandler handler = DiscardHandler.replaceOn(cache(NYC, 1));
      final FailureXSiteConsumer failureXSiteConsumer = FailureXSiteConsumer.replaceOn(cache(NYC, 0));
      final CounterBackupReceiverRepository counterRepository = CounterBackupReceiverRepository.replaceOn(cache(NYC, 0).getCacheManager());

      failureXSiteConsumer.fail(3);

      cache(LON, 0).put(key, VALUE);

      startStateTransfer(cache(LON, 0), NYC);
      assertOnline(LON, NYC);

      eventually(new Condition() {
         @Override
         public boolean isSatisfied() throws Exception {
            return handler.discarded;
         }
      });

      triggerTopologyChange(NYC, 1).get();

      awaitXSiteStateSent(LON);
      awaitXSiteStateReceived(NYC);

      AssertJUnit.assertEquals(0, failureXSiteConsumer.remainingFails());

      AssertJUnit.assertEquals(1, counterRepository.counter.get());

      assertInSite(NYC, new AssertCondition<Object, Object>() {
         @Override
         public void assertInCache(Cache<Object, Object> cache) {
            AssertJUnit.assertEquals(VALUE, cache.get(key));
         }
      });
   }

   /**
    * Simple scenario where the primary owner leaves the cluster and the NYC site master will apply the data locally
    * (test retry in the LON site).
    */
   public void testFailRetryLocally() throws ExecutionException, InterruptedException {
      takeSiteOffline(LON, NYC);
      final Object key = new MagicKey(cache(NYC, 1));
      final DiscardHandler handler = DiscardHandler.replaceOn(cache(NYC, 1));
      final FailureXSiteConsumer failureXSiteConsumer = FailureXSiteConsumer.replaceOn(cache(NYC, 0));
      final CounterBackupReceiverRepository counterRepository = CounterBackupReceiverRepository.replaceOn(cache(NYC, 0).getCacheManager());

      failureXSiteConsumer.failAlways();

      cache(LON, 0).put(key, VALUE);

      startStateTransfer(cache(LON, 0), NYC);
      assertOnline(LON, NYC);

      eventually(new Condition() {
         @Override
         public boolean isSatisfied() throws Exception {
            return handler.discarded;
         }
      });

      triggerTopologyChange(NYC, 1).get();

      awaitXSiteStateSent(LON);
      awaitXSiteStateReceived(NYC);

      //tricky part. When the primary owners dies, the site master or the other node can become the primary owner
      //if the site master is enabled, it will never be able to apply the state (XSiteStateConsumer is throwing exception!)
      //otherwise, the other node will apply the state
      if (XSiteStateTransferManager.STATUS_ERROR.equals(getXSitePushStatus(LON, NYC))) {
         AssertJUnit.assertEquals(3 /*max_retry + 1*/, counterRepository.counter.get());

         assertInSite(NYC, new AssertCondition<Object, Object>() {
            @Override
            public void assertInCache(Cache<Object, Object> cache) {
               AssertJUnit.assertNull(cache.get(key));
            }
         });
      } else {
         AssertJUnit.assertEquals(2 /*the 1st retry succeed*/, counterRepository.counter.get());
         assertInSite(NYC, new AssertCondition<Object, Object>() {
            @Override
            public void assertInCache(Cache<Object, Object> cache) {
               AssertJUnit.assertEquals(VALUE, cache.get(key));
            }
         });
      }
   }

   @Override
   protected void adaptLONConfiguration(BackupConfigurationBuilder builder) {
      super.adaptLONConfiguration(builder);
      builder.stateTransfer().maxRetries(2).waitTime(1000);
   }

   private static class CounterBackupReceiverRepository extends BackupReceiverRepositoryDelegator {

      private final AtomicInteger counter;

      private CounterBackupReceiverRepository(BackupReceiverRepository delegate) {
         super(delegate);
         this.counter = new AtomicInteger();
      }

      @Override
      public BackupReceiver getBackupReceiver(String originSiteName, String cacheName) {
         return new BackupReceiverDelegator(super.getBackupReceiver(originSiteName, cacheName)) {
            @Override
            public void handleStateTransferState(XSiteStatePushCommand cmd) throws Exception {
               counter.getAndIncrement();
               super.handleStateTransferState(cmd);
            }
         };
      }

      public static CounterBackupReceiverRepository replaceOn(CacheContainer cacheContainer) {
         BackupReceiverRepository delegate = extractGlobalComponent(cacheContainer, BackupReceiverRepository.class);
         CounterBackupReceiverRepository wrapper = new CounterBackupReceiverRepository(delegate);
         replaceComponent(cacheContainer, BackupReceiverRepository.class, wrapper, true);
         JGroupsTransport t = (JGroupsTransport) extractGlobalComponent(cacheContainer, Transport.class);
         CommandAwareRpcDispatcher card = t.getCommandAwareRpcDispatcher();
         replaceField(wrapper, "backupReceiverRepository", card, CommandAwareRpcDispatcher.class);
         return wrapper;
      }
   }

   private static class FailureXSiteConsumer implements XSiteStateConsumer {

      public static int FAIL_FOR_EVER = -1;
      private final XSiteStateConsumer delegate;
      //fail if > 0
      private int nFailures = 0;

      private FailureXSiteConsumer(XSiteStateConsumer delegate) {
         this.delegate = delegate;
      }

      @Override
      public void startStateTransfer(String sendingSite) {
         delegate.startStateTransfer(sendingSite);
      }

      @Override
      public void endStateTransfer(String sendingSite) {
         delegate.endStateTransfer(sendingSite);
      }

      @Override
      public void applyState(XSiteState[] chunk) throws Exception {
         boolean fail;
         synchronized (this) {
            fail = nFailures == FAIL_FOR_EVER;
            if (nFailures > 0) {
               fail = true;
               nFailures--;
            }
         }
         if (fail) {
            throw new CacheException("Induced Fail");
         }
         delegate.applyState(chunk);
      }

      @Override
      public String getSendingSiteName() {
         return delegate.getSendingSiteName();
      }

      public void fail(int nTimes) {
         if (nTimes < 0) {
            throw new IllegalArgumentException("nTimes should greater than zero but it is " + nTimes);
         }
         synchronized (this) {
            this.nFailures = nTimes;
         }
      }

      public void failAlways() {
         synchronized (this) {
            this.nFailures = FAIL_FOR_EVER;
         }
      }

      public int remainingFails() {
         synchronized (this) {
            return nFailures;
         }
      }

      public static FailureXSiteConsumer replaceOn(Cache<?, ?> cache) {
         return wrapComponent(cache, XSiteStateConsumer.class, new WrapFactory<XSiteStateConsumer, FailureXSiteConsumer, Cache<?, ?>>() {
            @Override
            public FailureXSiteConsumer wrap(Cache<?, ?> wrapOn, XSiteStateConsumer current) {
               return new FailureXSiteConsumer(current);
            }
         }, true);
      }
   }

   private static class DiscardHandler implements InboundInvocationHandler {

      private volatile boolean discarded = false;
      private final InboundInvocationHandler delegate;

      private DiscardHandler(InboundInvocationHandler delegate) {
         this.delegate = delegate;
      }

      public static DiscardHandler replaceOn(Cache<?, ?> cache) {
         return wrapGlobalComponent(cache.getCacheManager(), InboundInvocationHandler.class,
                                    new WrapFactory<InboundInvocationHandler, DiscardHandler, CacheContainer>() {
            @Override
            public DiscardHandler wrap(CacheContainer wrapOn, InboundInvocationHandler current) {
               DiscardHandler handler = new DiscardHandler(current);
               JGroupsTransport t = (JGroupsTransport) extractGlobalComponent(wrapOn, Transport.class);
               CommandAwareRpcDispatcher card = t.getCommandAwareRpcDispatcher();
               replaceField(handler, "inboundInvocationHandler", card, CommandAwareRpcDispatcher.class);
               return handler;
            }
         }, true);
      }

      @Override
      public void handle(CacheRpcCommand command, Address origin, Response response, boolean preserveOrder) throws Throwable {
         if (beforeHandle(command)) {
            delegate.handle(command, origin, response, preserveOrder);
         }
      }

      private boolean beforeHandle(CacheRpcCommand command) {
         if (!discarded) {
            discarded = command instanceof XSiteStatePushCommand;
         }
         return !discarded;
      }
   }

   private static class FailureHandler implements InboundInvocationHandler {

      public static int FAIL_FOR_EVER = -1;

      private final InboundInvocationHandler delegate;
      //fail if > 0
      private int nFailures = 0;

      private FailureHandler(InboundInvocationHandler delegate) {
         this.delegate = delegate;
      }

      public void fail(int nTimes) {
         if (nTimes < 0) {
            throw new IllegalArgumentException("nTimes should greater than zero but it is " + nTimes);
         }
         synchronized (this) {
            this.nFailures = nTimes;
         }
      }

      public void failAlways() {
         synchronized (this) {
            this.nFailures = FAIL_FOR_EVER;
         }
      }

      public int remainingFails() {
         synchronized (this) {
            return nFailures;
         }
      }

      public static FailureHandler replaceOn(final Cache<?, ?> cache) {
         return wrapGlobalComponent(cache.getCacheManager(), InboundInvocationHandler.class,
                                    new WrapFactory<InboundInvocationHandler, FailureHandler, CacheContainer>() {
            @Override
            public FailureHandler wrap(CacheContainer wrapOn, InboundInvocationHandler current) {
               FailureHandler handler = new FailureHandler(current);
               JGroupsTransport t = (JGroupsTransport) extractGlobalComponent(wrapOn, Transport.class);
               CommandAwareRpcDispatcher card = t.getCommandAwareRpcDispatcher();
               replaceField(handler, "inboundInvocationHandler", card, CommandAwareRpcDispatcher.class);
               return handler;
            }
         }, true);
      }

      @Override
      public void handle(CacheRpcCommand command, Address origin, Response response, boolean preserveOrder) throws Throwable {
         if (beforeHandle(command, response)) {
            delegate.handle(command, origin, response, preserveOrder);
         }
      }

      private synchronized boolean beforeHandle(CacheRpcCommand command, Response response) {
         if (command instanceof XSiteStatePushCommand) {
            boolean fail;
            synchronized (this) {
               fail = nFailures == FAIL_FOR_EVER;
               if (nFailures > 0) {
                  fail = true;
                  nFailures--;
               }
            }
            if (fail) {
               response.send(new ExceptionResponse(new CacheException("Induced Fail.")), false);
               return false;
            }
         }
         return true;
      }
   }


}
