/*******************************************************************************
 * Copyright (c) 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.oauth.core.internal.oauth20.token.impl;

import java.util.Map;
import java.util.Set;

import com.ibm.oauth.core.api.oauth20.token.OAuth20Token;
import com.ibm.oauth.core.internal.oauth20.OAuth20Constants;

/**
 * This class is the base class for all token objects (access tokens and grants)
 * that will be generated by TFIM. Because they may be cached in the DMAP token
 * cache implementation, they should implement ImmutableObject. This base class
 * defines those methods as returning "this" because there are no setters on the
 * class. If a subclass providers modification methods (like setters) it should
 * override the ImmutableObject methods with versions that return an immutable
 * version of the class.
 *
 */
public abstract class OAuth20TokenBase implements OAuth20Token {

    /**
     * A unique key to identity this token in a cache
     */
    private String _id = null;

    /**
     * The component instance id in which this token is used
     */
    private String _componentId = null;

    /**
     * The token type i.e. "AUTHORIZATION_CODE" or "ACCESS_TOKEN"
     */
    private String _type = null;

    /**
     * The UTC time (milliseconds) that this token was created at
     */
    private long _createdAt = 0;

    /**
     * The UTC time (milliseconds) that this token was last retrieved from cache
     */
    private long _lastAccess = 0;

    /**
     * The state id of this token
     */
    private String _stateId = null;

    /**
     * The lifetime (seconds) of this token
     */
    private int _lifetimeSeconds = 0;

    /**
     * TRUE if this token should be cached, FALSE otherwise
     */
    private boolean _isPersistent;

    /**
     * The type of grant for an access token
     */
    private String _grantType = null;

    /**
     * The extension properties map for this OAuth20Token
     */
    private Map<String, String[]> _extensionProperties = null;

    public OAuth20TokenBase(String id, String componentId, String type,
            String stateId, int lifetimeSeconds, boolean isPersistent,
            Map<String, String[]> extensionProperties, String grantType) {
        _id = id;
        init(componentId, type, stateId, lifetimeSeconds, isPersistent,
                extensionProperties, grantType);
    }

    private void init(String componentId, String type, String stateId,
            int lifetimeSeconds, boolean isPersistent,
            Map<String, String[]> extensionProperties, String grantType) {
        _componentId = componentId;
        _type = type;
        _createdAt = System.currentTimeMillis();
        _stateId = stateId;
        _lifetimeSeconds = lifetimeSeconds;
        _isPersistent = isPersistent;
        _extensionProperties = extensionProperties;
        _grantType = grantType;
    }

    /**
     * @return the unique key to identity this token in a cache
     */
    @Override
    public String getId() {
        return _id;
    }

    /**
     * @return the componentId to which this token applies
     */
    @Override
    public String getComponentId() {
        return _componentId;
    }

    /**
     * @return the token type i.e. "AUTHORIZATION_CODE" or "ACCESS_TOKEN"
     */
    @Override
    public String getType() {
        return _type;
    }

    /**
     * @return The timestamp in milliseconds since the epoch when this token was
     *         created. This can be used along with the lifetime to calculate an
     *         expiry time.
     */
    @Override
    public long getCreatedAt() {
        return _createdAt;
    }

    /**
     * @return the state id of this token
     */
    @Override
    public String getStateId() {
        return _stateId;
    }

    /**
     * @return the lifetime (seconds) of this token
     */
    @Override
    public int getLifetimeSeconds() {
        return _lifetimeSeconds;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public boolean isPersistent() {
        return _isPersistent;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getGrantType() {
        return _grantType;
    }

    /**
     * @return an array of property names which can be used in calls to
     *         getExtensionProperty() to obtain extension property values for a
     *         particular token. Different extension properties are be set by
     *         different token spec implementations. The Bearer token spec does
     *         not use extension properties.
     */
    @Override
    public String[] getExtensionPropertyNames() {
        String[] result = null;
        if (_extensionProperties != null && _extensionProperties.size() > 0) {
            Set<String> keys = _extensionProperties.keySet();
            if (keys != null) {
                result = keys.toArray(new String[keys.size()]);
            }
        }
        return result;
    }

    /**
     * @return an array of properties for a given property name. Different
     *         extension properties are be set by different token spec
     *         implementations. The Bearer token spec does not use extension
     *         properties.
     */
    @Override
    public String[] getExtensionProperty(String propertyName) {
        String[] result = null;
        if (_extensionProperties != null && _extensionProperties.size() > 0) {
            result = _extensionProperties.get(propertyName);
        }
        return result;
    }

    /**
     * @return the extensionProperties
     */
    @Override
    public Map<String, String[]> getExtensionProperties() {
        return _extensionProperties;
    }

    // these aren't used here but are needed to comply with the interface changes for app-password and app-token
    /** {@inheritDoc} */
    @Override
    public String getUsedFor() {
        return null;
    }

    /** {@inheritDoc} */
    @Override
    public String[] getUsedBy() {
        return null;
    }

    /** {@inheritDoc} */
    @Override
    public String getAppName() {
        return null;
    }

    /** {@inheritDoc} */
    @Override
    public long getLastAccess() {
        return _lastAccess;
    }

    /** {@inheritDoc} */
    @Override
    public void setLastAccess() {
        _lastAccess = System.currentTimeMillis();
    }
    
    /* (non-Javadoc)
     * @see com.ibm.oauth.core.api.oauth20.token.OAuth20Token#getCodeChallenge()
     */
    @Override
    public String getCodeChallenge() {  
        String[] buf = getExtensionProperty(OAuth20Constants.EXTERNAL_CLAIMS_PREFIX + OAuth20Constants.CODE_CHALLENGE);
        return buf == null ? null : buf[0];
    }

    /* (non-Javadoc)
     * @see com.ibm.oauth.core.api.oauth20.token.OAuth20Token#getCodeChallengeMethod()
     */
    @Override
    public String getCodeChallengeMethod() {
        String[] buf = getExtensionProperty(OAuth20Constants.EXTERNAL_CLAIMS_PREFIX + OAuth20Constants.CODE_CHALLENGE_METHOD);
        return buf == null ? null : buf[0];
    }

    /**
     * @return a string representation of this token
     */
    @Override
    public String toString() {
        StringBuffer token = new StringBuffer();

        token.append("id=" + _id + "&");
        token.append("component_id=" + _componentId + "&");
        token.append("type=" + _type + "&");
        token.append("created_at=" + _createdAt + "&");
        token.append("state_id=" + _stateId + "&");
        token.append("lifetime_seconds=" + _lifetimeSeconds + "&");
        token.append("is_persistent=" + _isPersistent + "&");
        token.append("grant_type=" + _grantType);

        return token.toString();
    }
}
