/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.ws;

import org.jboss.soa.dsp.server.ServerConfig;

import java.io.*;

public class DeploymentBuilder
{
  private static final String DSP = "dsp";
  
  private File war;
  private File webInf;
  private File wsdlDir;
  private File wsdl;
  private String endpointId;
  
  private javax.xml.ws.Provider<?> provider;

  private ServerConfig serverConfig;

  public DeploymentBuilder(ServerConfig serverConfig)
  {
    this.serverConfig = serverConfig;
  }

  public DeploymentBuilder setEndpoint(String endpointId)
  {
    File tmpDir = new File(serverConfig.getServerTempDir(), DSP);

    tmpDir.mkdir();
    File fakeWebApp = new File(tmpDir, DSP+"-"+endpointId+".war");
    File fakeWebInf = new File(fakeWebApp, "WEB-INF");
    fakeWebInf.mkdirs();

    // Workaround
    // See https://jira.jboss.org/jira/browse/JBWS-2718
    File fakeWSDLDir = new File(fakeWebInf, "wsdl");
    fakeWSDLDir.mkdirs();

    this.war = fakeWebApp;
    this.webInf = fakeWebInf;
    this.wsdlDir = fakeWSDLDir;
    this.endpointId = endpointId;

    return this;
  }
  
  public File getWar() {
	  return(this.war);
  }
  
  public File getWebInf() {
	  return(this.webInf);
  }
  
  public File getWSDLDir() {
	  return(this.wsdlDir);
  }
  
  public String getEndpoint() {
	  return(this.endpointId);
  }

  public DeploymentBuilder setWSDL(File wsdl, File root)
  {
	this.wsdl = wsdl;
	
    copy(wsdl, new File(this.wsdlDir, wsdl.getName()));

    // any related artifact as well (brute force, I know)
    //File parent = wsdl.getParentFile();
    //assert parent.isDirectory();

    FileFilter filter = new FileFilter()
    {
      public boolean accept(File f)
      {
    	  // Deploy all artifacts, as we don't know what might be relevant in a
    	  // particular execution context
        return (true);
      }
    };

    for(File f : root.listFiles(filter))
    {
      if(f.equals(wsdl)) continue;
      copy(f, new File(this.wsdlDir, f.getName()));
      copy(f, new File(this.webInf, f.getName()));
      
      
/* Causes problems with testcase RIFTSAW-70 on jbossws-native. Ok
	on jbossws-cxf. So just copying all artifacts to both locations
	for now.

      if (f.getName().endsWith(".wsdl") ||
    		  f.getName().endsWith(".xsd")) {
    	  copy(f, new File(this.wsdlDir, f.getName()));
      } else {
    	  copy(f, new File(this.webInf, f.getName()));
     }
*/
    }
    return this;
  }
  
  public File getWSDL() {
	  return(this.wsdl);
  }
  
  public DeploymentBuilder setProvider(javax.xml.ws.Provider<?> provider) {
	  this.provider = provider;
	  
	  return(this);
  }
  
  public javax.xml.ws.Provider<?> getProvider() {
	  return(this.provider);
  }
  
  public DeploymentBuilder process(BuildProcessor processor) {
	  if (processor != null) {
		  processor.process(this);
	  }
	  
	  return(this);
  }

  public File build()
  {
    return this.war;
  }

  public static synchronized void copy(File src, File dest)
  {
    InputStream in = null;
    OutputStream out = null;

    try
    {
      if (src.isDirectory())
      {
        if (!dest.exists()) {
          dest.mkdir();
        }

        String[] children = src.list();
        for (int i=0; i<children.length; i++) {
          copy(new File(src, children[i]),
              new File(dest, children[i]));
        }
      } else {

        in = new FileInputStream(src);
        out = new FileOutputStream(dest);

        // Copy the bits from instream to outstream
        byte[] buf = new byte[1024];
        int len;
        while ((len = in.read(buf)) > 0) {
          out.write(buf, 0, len);
        }
        in.close();
        out.close();
      }

    }
    catch (IOException e)
    {
      throw new RuntimeException("Failed to copy files", e);
    }
    finally
    {
      try
      {
        if(in!=null) in.close();
      }
      catch (IOException e)
      {
        //
      }

      try
      {
        if(out!=null) out.close();
      }
      catch (IOException e)
      {
        //
      }
    }
  }
}
