/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.ws;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
//import org.apache.xerces.jaxp.DocumentBuilderFactoryImpl;
import org.jboss.soa.dsp.MessageAdapter;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.wsdl.*;
import javax.wsdl.extensions.ElementExtensible;
import javax.wsdl.extensions.ExtensibilityElement;
import javax.wsdl.extensions.http.HTTPBinding;
import javax.wsdl.extensions.soap.SOAPBinding;
import javax.wsdl.extensions.soap.SOAPHeader;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.soap.*;

import java.util.*;

/**
 * Adopts {@link javax.xml.soap.SOAPMessage}'s to ODE's internal
 * {@link org.apache.ode.bpel.iapi.Message} representation and vice versa.
 *
 * @see org.jboss.soa.bpel.runtime.ws.WebServiceClient
 *
 * @author Heiko.Braun <heiko.braun@jboss.com>
 */
public class SOAPMessageAdapter
{
  protected static final Log log = LogFactory.getLog(SOAPMessageAdapter.class);

  private Definition wsdl;
  private QName serviceName;

  private String portName;
  private Service serviceDef;
  private Binding binding;

  private Port port;
  private boolean isRPC;

  private SOAPBinding soapBinding;

  private SOAPFactory soapFactory;

  public SOAPMessageAdapter(Definition wsdl, QName serviceName, String portName)
  {
    this.wsdl = wsdl;
    this.serviceName = serviceName;
    this.portName = portName;

    serviceDef = wsdl.getService(serviceName);
    if (serviceDef == null)
      throw new RuntimeException("Service not found "+serviceName);

    port = serviceDef.getPort(portName);
    if (port == null)
      throw new RuntimeException("Port '"+portName+"' not found on service: "+serviceName);

    binding = port.getBinding();
    if (binding == null)
      throw new RuntimeException("No binding for port "+portName);

    if (!useSOAPBinding(port)) {
      throw new RuntimeException("No SOAP binding for port"+portName);
    }
    soapBinding = (SOAPBinding)getBindingExtension(port);


    String style = soapBinding.getStyle();
    isRPC = style != null && style.equals("rpc");

    try
    {
      this.soapFactory = SOAPFactory.newInstance();
    }
    catch (SOAPException e)
    {
      throw new RuntimeException(e);
    }
  }

  /**
   * This method creates the SOAP request and returns the SOAPAction field.
   * 
   * @param soapMessage
   * @param odeRequestMessage
   * @param wsdlOperation
   * @return The SOAP action
   */
  public String createSoapRequest(SOAPMessage soapMessage, MessageAdapter odeRequestMessage, Operation wsdlOperation) 
  {
	  String ret=null;
	  
    BindingOperation bop = binding.getBindingOperation(wsdlOperation.getName(), null, null);
    if (bop == null)
      throw new RuntimeException("Operation "+wsdlOperation.getName()+"not found on "+serviceName+"/"+portName);

    BindingInput bi = bop.getBindingInput();
    if (bi == null)
      throw new RuntimeException("Binding input not found on "+serviceName+"/"+portName);

    // Headers
    createSoapHeaders(
        soapMessage,
        getSOAPHeaders(bi),
        wsdlOperation.getInput().getMessage(),
        odeRequestMessage.getHeaderParts(),
        odeRequestMessage.getMessage()
    );


    // SOAP Body
    javax.wsdl.extensions.soap.SOAPBody wsdlSoapBody = getSOAPBody(bi);

    createSoapBody(
        soapMessage,
        wsdlSoapBody,
        wsdlOperation.getInput().getMessage(),
        odeRequestMessage.getMessage(),
        wsdlOperation.getName()
    );

    // Discover SOAPAction
    for (Object extension : bop.getExtensibilityElements()) {
    	if (extension instanceof javax.wsdl.extensions.soap.SOAPOperation) {
    		javax.wsdl.extensions.soap.SOAPOperation soapop=
    				(javax.wsdl.extensions.soap.SOAPOperation)extension;
    		
    		if (soapop.getSoapActionURI() != null) { 	
    			ret = soapop.getSoapActionURI();
    			break;
    		}
    	}
    }
    
    return(ret);
  }

  public boolean isRPC()
  {
    return isRPC;
  }

  public void createSoapResponse(SOAPMessage soapMessage, MessageAdapter odeResponseMessage, Operation wsdlOperation)
  {

    BindingOperation bop = binding.getBindingOperation(wsdlOperation.getName(),null,null);
    if (bop == null)
      throw new RuntimeException("Operation "+wsdlOperation.getName()+"not found on "+serviceName+"/"+portName);

    BindingOutput bo = bop.getBindingOutput();
    if (bo == null)
      throw new RuntimeException("Binding output not found on "+serviceName+"/"+portName);

    // Headers
    if (odeResponseMessage.getHeaderParts().size() > 0 || getSOAPHeaders(bo).size() > 0)
      createSoapHeaders(
          soapMessage,
          getSOAPHeaders(bo),
          wsdlOperation.getOutput().getMessage(),
          odeResponseMessage.getHeaderParts(),
          odeResponseMessage.getMessage()
      );


    // SOAP Body
    javax.wsdl.extensions.soap.SOAPBody wsdlSOAPBody = getSOAPBody(bo);
    createSoapBody(
        soapMessage,
        wsdlSOAPBody,
        wsdlOperation.getOutput().getMessage(),
        odeResponseMessage.getMessage(),
        wsdlOperation.getName() + "Response"
    );

  }

  private void createSoapBody(SOAPMessage soapMessage,
                              javax.wsdl.extensions.soap.SOAPBody wsdlSoapBody, javax.wsdl.Message wsdlMessageDef,
                              Element message, String operationName)
  {
    try
    {
      SOAPBody soapBody = soapMessage.getSOAPBody();

      SOAPElement partHolder = null;
      if(isRPC)
      {
        partHolder = soapFactory.createElement(new QName(wsdlSoapBody.getNamespaceURI(), operationName, "odens"));
      }
      else
      {
        partHolder = soapBody;
      }

      List<Part> parts = wsdlMessageDef.getOrderedParts(wsdlSoapBody.getParts());
      for(Part part : parts)
      {
        Element srcPartEl = findChildByName(message, new QName(null, part.getName()));
        if (srcPartEl == null)
          throw new RuntimeException("Part is missing: " +part.getName());

        SOAPElement partElement = soapFactory.createElement(srcPartEl);
        if (isRPC)
        {
          partHolder.addChildElement(partElement);
        }
        else
        {
          for (Iterator<SOAPElement> i = partElement.getChildElements(); i.hasNext();) partHolder.addChildElement(i.next());
        }
      }

      // late bind
      if(isRPC)
        soapBody.addChildElement(partHolder);
    }
    catch (SOAPException e)
    {
      throw new RuntimeException("Failed to create soap body",e);
    }
  }

  private void createSoapHeaders(SOAPMessage soapMessage, List<SOAPHeader> headers,
                                 javax.wsdl.Message wsdlMessageDef,
                                 Map<String, Node> headerParts, Element message)
  {
	  try {
		  javax.xml.soap.SOAPHeader soapHeader = soapMessage.getSOAPHeader();
		  if (soapHeader==null) soapHeader = soapMessage.getSOAPPart().getEnvelope().addHeader();
		  for (Node headerNode : headerParts.values()) {
			  //like we may have password header with null value.
			  if(headerNode == null){
				  continue;
			  }
			  if (Node.ELEMENT_NODE == headerNode.getNodeType()) {
				  if (getFirstChildWithName(new QName(headerNode.getNamespaceURI(), headerNode.getLocalName()),soapHeader) == null) {
					  SOAPElement partElement = soapFactory.createElement((Element) headerNode);
					  soapHeader.addChildElement(partElement);
				  }
			  } else {
				  throw new RuntimeException("SOAP header must be a node_element " + headerNode);
			  }
		  }
		  
		  //Add soap header according to binding.
		  for (SOAPHeader header : headers) {
			  Element headerEl = findChildByName(message, new QName(null, header.getPart()));
			  if (headerEl != null) {	
				  
				  // RIFTSAW-305 - don't think the part name should be added to the SOAP header
				  //SOAPElement soapHeaderEl = soapFactory.createElement(new QName(header.getMessage().getNamespaceURI(), header.getPart(),"odens"));
				  NodeList list = headerEl.getChildNodes();
				  for(int i=0; i< list.getLength(); i++) {
					  SOAPElement partElement = soapFactory.createElement((Element)list.item(i));
					  soapHeader.addChildElement(partElement);
				  }
				  //soapHeader.addChildElement(soapHeaderEl);
			  }
		  }
	  } catch (SOAPException e) {
		  throw new RuntimeException("Failed to create soap header",e);
	  }
  }

  public void parseSoapResponse(MessageAdapter odeMessage,
                                SOAPMessage soapMessage, javax.wsdl.Operation odeOperation) {
    BindingOperation bop = binding.getBindingOperation(odeOperation.getName(), null, null);
    if (bop == null)
      throw new RuntimeException("Operation "+odeOperation.getName()+"not found on "+serviceName+"/"+portName);

    BindingOutput bo = bop.getBindingOutput();
    if (bo == null)
      throw new RuntimeException("Binding output not found on "+serviceName+"/"+portName);

    extractSoapBodyParts(odeMessage, soapMessage, getSOAPBody(bo), odeOperation.getOutput().getMessage(), odeOperation.getName() + "Response");
    extractSoapHeaderParts(odeMessage, soapMessage, getSOAPHeaders(bo), odeOperation.getOutput().getMessage());
  }

  public void parseSoapRequest(MessageAdapter odeMessage,
      SOAPMessage soapMessage,
      Operation op)
  {

    BindingOperation bop = binding.getBindingOperation(op.getName(), null, null);

    if (bop == null)
      throw new RuntimeException("Binding operation not found ("+serviceName+"/"+portName);

    BindingInput bi = bop.getBindingInput();
    if (bi == null)
      throw new RuntimeException("Binding input not found"+serviceName+"/"+portName);

    extractSoapBodyParts(odeMessage, soapMessage, getSOAPBody(bi), op.getInput().getMessage(), op.getName());
    extractSoapHeaderParts(odeMessage, soapMessage, getSOAPHeaders(bi), op.getInput().getMessage());
  }

  public void createSoapFault(SOAPMessage soapMessage, Element message, QName faultName, Operation op)
  {
    try
    {
      Element detail = buildSoapDetail(message, faultName, op);
      SOAPFault fault = soapMessage.getSOAPBody().addFault();
      fault.setFaultCode(faultName);
      if(detail!=null)
        fault.addDetail().addChildElement(soapFactory.createElement(detail));      
    }
    catch (SOAPException e)
    {
      throw new RuntimeException("Failed to create fault", e);
    }
  }

  private Element buildSoapDetail(Element message, QName faultName, Operation op)
  {
    if (faultName.getNamespaceURI() == null)
      return toFaultDetail(faultName, message);
    if (op == null) {
      return toFaultDetail(faultName, message);
    }
    Fault f = op.getFault(faultName.getLocalPart());
    if (f == null)
      return toFaultDetail(faultName, message);

    // For faults, there will be exactly one part.
    Part p = (Part)f.getMessage().getParts().values().iterator().next();
    if (p == null)
      return toFaultDetail(faultName, message);
    Element partEl= findChildByName(message,new QName(null,p.getName()));
    if (partEl == null)
      return toFaultDetail(faultName, message);
    Element detail = findChildByName(partEl, p.getElementName());
    if (detail == null)
      return toFaultDetail(faultName, message);

    return detail;
  }

  private Element toFaultDetail(QName fault, Element message) {
    if (message == null) return null;
    Element firstPart = getFirstChildElement(message);
    if (firstPart == null) return null;
    Element detail = getFirstChildElement(firstPart);
    if (detail == null) return firstPart;
    return detail;
  }

  private void extractSoapHeaderParts(MessageAdapter odeMessage, SOAPMessage soapMessage, List<SOAPHeader> headerDefs,javax.wsdl.Message wsdlMessageDef)
  {
	  try {
		  javax.xml.soap.SOAPHeader soapHeader = soapMessage.getSOAPHeader();
		  // Checking that the definitions we have are at least there
	      for (SOAPHeader headerDef : headerDefs)
	          handleSoapHeaderPartDef(odeMessage, soapHeader, headerDef, wsdlMessageDef);
	
	      // Extracting whatever header elements we find in the message, binding and abstract parts
	      // aren't reliable enough given what people do out there.
	      if (soapHeader != null) {
	      	Iterator headersIter = soapHeader.getChildElements();
	      	while (headersIter.hasNext()) {
	      		Object obj=headersIter.next();

	      		// Should be SOAPHeaderElement, but CXF also returns javax.xml.soap.Text
	      		// objects aswell
	      		if (obj instanceof javax.xml.soap.SOAPHeaderElement) {
		      		javax.xml.soap.SOAPHeaderElement headerElem = (javax.xml.soap.SOAPHeaderElement) obj;
		      		String partName = findHeaderPartName(headerDefs, headerElem.getElementQName());
		      		Document doc = newDocument();
	
		      		// RIFTSAW-74 - slight modification to avoid jbossws exception when reconstructing the
		      		// SOAP message.
	
		      		//Element destPart = doc.createElementNS(null, partName);
		      		//destPart.appendChild(doc.importNode(headerElem, true));
		      		//odeMessage.setHeaderPart(partName, destPart);
		      		odeMessage.setHeaderPart(partName, (Element)doc.importNode(headerElem, true));
	      		}
	      	}
	      }
	  }
	    catch (SOAPException e)
	    {
	      throw new RuntimeException("Failed to extracts header parts",e);
	    }
  }
  
  private String findHeaderPartName(List<SOAPHeader> headerDefs, QName elmtName) {
      for (SOAPHeader headerDef : headerDefs) {
    	  javax.wsdl.Message hdrMsg = wsdl.getMessage(headerDef.getMessage());
          for (Object o : hdrMsg.getParts().values()) {
              Part p = (Part) o;
              if (p.getElementName() != null &&
            		  p.getElementName().equals(elmtName)) return p.getName();
          }
      }
      return elmtName.getLocalPart();
  }
  
  private void handleSoapHeaderPartDef(Object odeMessage, javax.xml.soap.SOAPHeader header, SOAPHeader headerdef,
		  javax.wsdl.Message msgType)  {
      // Is this header part of the "payload" messsage?
      boolean payloadMessageHeader = headerdef.getMessage() == null || headerdef.getMessage().equals(msgType.getQName());
      boolean requiredHeader = payloadMessageHeader || Boolean.TRUE.equals(headerdef.getRequired());

      if (header == null) {
      	if (requiredHeader)
      		throw new RuntimeException("Soap Header is missing a required field " + headerdef.getElementType());

      	return;
      }

      javax.wsdl.Message hdrMsg = wsdl.getMessage(headerdef.getMessage());
      if (hdrMsg == null)
          return;
      Part p = hdrMsg.getPart(headerdef.getPart());
      if (p == null || p.getElementName() == null)
          return;

      SOAPElement headerEl = getFirstChildWithName(p.getElementName(), header);
      if (requiredHeader && headerEl == null)
          throw new RuntimeException("Soap Header is missing a required field " + headerdef.getElementType());

      if (headerEl == null) return;

      /* RIFTSAW-127 - this was duplicating the header part in the consolidated (merged) message stored in the
       * BPEL process - but this code actually causes two levels of 'part' to be defined - e.g.
       * <conversionId><conversationId><details .... /></conversatioId></conversationId>
      Document doc = DOMUtils.newDocument();
      Element destPart = doc.createElementNS(null, p.getName());
      destPart.appendChild(doc.importNode(headerEl, true));
      odeMessage.setHeaderPart(p.getName(), destPart);
      */
  }

  private void extractSoapBodyParts(
      MessageAdapter odeMessage,
      SOAPMessage soapMessage,
      javax.wsdl.extensions.soap.SOAPBody wsdlSOAPBody,
      javax.wsdl.Message wsdlMessageDef, String operationName)
  {
    try
    {
      SOAPBody soapBody = soapMessage.getSOAPBody();
      List<Part> parts = wsdlMessageDef.getOrderedParts(wsdlSOAPBody.getParts());

      if(isRPC)
      {
        // In RPC the body element is the operation name, wrapping parts. Order doesn't really matter as far as
        // we're concerned. All we need to do is copy the soap:body children, since doc-lit rpc looks the same
        // in ode and soap.

        QName rpcWrapQName = new QName(wsdlSOAPBody.getNamespaceURI(), operationName);
        SOAPElement partWrapper = getFirstChildWithName(rpcWrapQName, soapBody);

        if (partWrapper == null)
          throw new RuntimeException("Expected part wrapper '"+rpcWrapQName+"'missing on service:"+serviceName+"/"+portName);

        for(Part part : parts)
        {
          Element srcPart = getFirstChildWithName(new QName(null, part.getName()), partWrapper);
          if (srcPart == null)
            throw new RuntimeException("Soap body does not contain required part +"+part.getName());

          odeMessage.setPart(srcPart.getLocalName(), srcPart);
        }
      }
      else
      {
        // In doc-literal style, we expect the elements in the body to correspond (in order)
        // to the parts defined in the binding.
        // All the parts should be element-typed, otherwise it is a mess.
        List<SOAPElement> childElements = new ArrayList<SOAPElement>();
        final Iterator children = soapBody.getChildElements() ;
        while(children.hasNext())
        {
          final Node node = (Node)children.next() ;
          if (node instanceof SOAPElement)
            childElements.add((SOAPElement)node);
        }

        Iterator<SOAPElement> srcParts = childElements.iterator();
        for(Part part : parts)
        {
          SOAPElement srcPart = srcParts.next();
          Document doc = newDocument();
          Element destPart = doc.createElementNS(null, part.getName());
          destPart.appendChild(doc.importNode(srcPart, true));
          odeMessage.setPart(part.getName(), destPart);
        }
      }
    }
    catch (SOAPException e)
    {
      throw new RuntimeException("Failed to extract soap body parts", e);
    }
  }

  private static SOAPElement getFirstChildWithName(QName name, SOAPElement parent)
  {
    SOAPElement match = null;
    Iterator iterator = parent.getChildElements(name);
    while(iterator.hasNext())
    {
      match= (SOAPElement)iterator.next();
    }
    return match;
  }

  /*private static Element cloneElement(Element source)
  {
    // TODO: https://jira.jboss.org/jira/browse/RIFTSAW-38
    // For now create a deep copy (performance hit)
    try
    {
      DocumentBuilder builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
      Document doc = builder.newDocument();
      return (Element)doc.importNode(source, true);
    }
    catch (ParserConfigurationException e)
    {
      throw new RuntimeException(e);
    }
  } */

  public static <T> T getFirstExtensibilityElement(ElementExtensible parent, Class<T> cls) {
    Collection<T> ee = filter(parent.getExtensibilityElements(), cls);

    return ee.isEmpty() ? null : ee.iterator().next();

  }

  public static javax.wsdl.extensions.soap.SOAPBody getSOAPBody(ElementExtensible ee) {
    return getFirstExtensibilityElement(ee, javax.wsdl.extensions.soap.SOAPBody.class);
  }

  public static List<SOAPHeader> getSOAPHeaders(ElementExtensible eee) {
    return filter(new ArrayList<SOAPHeader>(), (Collection<Object>) eee.getExtensibilityElements(),
        SOAPHeader.class);
  }

  public static Fault parseSoapFault(
      Element odeMessage,
      SOAPMessage soapMessage,
      javax.wsdl.Operation operation)
  {
    Fault fdef = null;
    try
    {
      SOAPFault flt = soapMessage.getSOAPBody().getFault();
      Detail detail = flt.getDetail();
      fdef = inferFault(operation, flt);
      if(fdef!=null)
      {
        Part pdef = (Part)fdef.getMessage().getParts().values().iterator().next();
        Element partel = odeMessage.getOwnerDocument().createElementNS(null,pdef.getName());
        odeMessage.appendChild(partel);

        Element childByName = findChildByName(detail, pdef.getElementName());
        if (childByName != null)
        {
          partel.appendChild(odeMessage.getOwnerDocument().importNode(childByName, true));
        }
        else
        {
          partel.appendChild(odeMessage.getOwnerDocument().importNode(detail,true));
        }
      }
    }
    catch (Exception e)
    {
      throw new RuntimeException("Failed to parse SOAP Fault",e);
    }

    return fdef;
  }

  public static Fault parseSoapFault(
	      Element odeMessage,
	      SOAPFault flt,
	      javax.wsdl.Operation operation)
  {
      Fault fault=inferFault(operation, flt);

      if(fault!=null)
      {
          Detail detail = flt.getDetail();
        Part pdef = (Part)fault.getMessage().getParts().values().iterator().next();
        Element partel = odeMessage.getOwnerDocument().createElementNS(null,pdef.getName());
        odeMessage.appendChild(partel);

        Element childByName = findChildByName(detail, pdef.getElementName());
        if (childByName != null)
        {
          partel.appendChild(odeMessage.getOwnerDocument().importNode(childByName, true));
        }
        else
        {
          partel.appendChild(odeMessage.getOwnerDocument().importNode(detail,true));
        }
      }

    return fault;
  }

  private static Fault inferFault(Operation operation, SOAPFault flt) {
    if (!flt.hasDetail())
      return null;
    // The detail is a dummy <detail> node containing the interesting fault element
    Element element = getFirstChildElement(flt.getDetail());
    
    if (element == null) {
    	return(null);
    }
    
    QName elName=new QName(element.getNamespaceURI(), element.getLocalName());
    return inferFault(operation, elName);
  }

  
  // Code taken from Apache ODE util module
  
  private static ThreadLocal<DocumentBuilder> __builders = new ThreadLocal();
  
  private static DocumentBuilderFactory __documentBuilderFactory ;

  static {
      initDocumentBuilderFactory();
  }

  /**
   * Initialize the document-builder factory.
   */
  private static void initDocumentBuilderFactory() {
      DocumentBuilderFactory f = javax.xml.parsers.DocumentBuilderFactory.newInstance();//new DocumentBuilderFactoryImpl();
      f.setNamespaceAware(true);
      __documentBuilderFactory = f;
  }

  private static DocumentBuilder getBuilder() {
      DocumentBuilder builder = __builders.get();
      if (builder == null) {
          synchronized (__documentBuilderFactory) {
              try {
                  builder = __documentBuilderFactory.newDocumentBuilder();
                  builder.setErrorHandler(new org.xml.sax.ErrorHandler() {

					public void error(SAXParseException arg0)
							throws SAXException {
						log.error("Parser error", arg0);
					}

					public void fatalError(SAXParseException arg0)
							throws SAXException {
						log.error("Parser fatal error", arg0);
					}

					public void warning(SAXParseException arg0)
							throws SAXException {
						log.warn("Parser warning", arg0);
					}
                	  
                  });
              } catch (ParserConfigurationException e) {
                  log.error(e);
                  throw new RuntimeException(e);
              }
          }
          __builders.set(builder);
      }
      return builder;
  }

  public static Document newDocument() {
      DocumentBuilder db = getBuilder();
      return db.newDocument();
  }

  /**
   * Return the first child element of the given element. Null if no children
   * are found.
   *
   * @param elem Element whose child is to be returned
   *
   * @return the first child element.
   */
  public static Element getFirstChildElement(Element elem) {
      return (Element) findChildByType(elem, Node.ELEMENT_NODE);
  }

  public static Element findChildByName(Element parent, QName name) {
      return findChildByName(parent, name, false);
  }

  public static Element findChildByName(Element parent, QName name, boolean recurse) {
      if (parent == null)
          throw new IllegalArgumentException("null parent");
      if (name == null)
          throw new IllegalArgumentException("null name");

      NodeList nl = parent.getChildNodes();
      for (int i = 0; i < nl.getLength(); ++i) {
          Node c = nl.item(i);
          if(c.getNodeType() != Node.ELEMENT_NODE)
              continue;
          // For a reason that I can't fathom, when using in-mem DAO we actually get elements with
          // no localname.
          String nodeName = c.getLocalName() != null ? c.getLocalName() : c.getNodeName();
          if (new QName(c.getNamespaceURI(),nodeName).equals(name))
              return (Element) c;
      }

      if(recurse){
          NodeList cnl = parent.getChildNodes();
          for (int i = 0; i < cnl.getLength(); ++i) {
              Node c = cnl.item(i);
              if(c.getNodeType() != Node.ELEMENT_NODE)
                  continue;
              Element result = findChildByName((Element)c, name, recurse);
              if(result != null)
                  return result;
          }
      }
      return null;
  }

  public static Node findChildByType(Element elem, int type) {
      if (elem == null)
          throw new NullPointerException("elem parameter must not be null!");

      for (Node n = elem.getFirstChild(); n != null; n = n.getNextSibling()) {
          if (n.getNodeType() == type) {
              return n;
          }
      }
      return null;
  }
  
  public static <C extends Collection<T>, S, T extends S> C filter(C newList, Iterator<S> iterator, Class<T> t) {
      while (iterator.hasNext()) {
          S next = iterator.next();
          if (t.isAssignableFrom(next.getClass())) {
              newList.add((T) next);
          }
      }
      return newList;
  }

  @SuppressWarnings("unchecked")
  public static <T> Collection<T> filter(Collection src, final Class<T> aClass) {
      return filter(new ArrayList<T>(src.size()), src.iterator(), aClass);
  }

  public static <C extends Collection<T>, S, T extends S> C filter(C dest, Collection<S> src, Class<T> t) {
      return filter(dest, src.iterator(), t);
  }
  
  @SuppressWarnings("unchecked")
  public static Fault inferFault(Operation operation, QName elName) {
      for (Fault f : (Collection<Fault>) operation.getFaults().values()) {
          if (f.getMessage() == null) continue;
          Collection<Part> parts = f.getMessage().getParts().values();
          if (parts.isEmpty()) continue;
          Part p = parts.iterator().next();
          if (p.getElementName() == null) continue;
          if (p.getElementName().equals(elName)) return f;
      }
      return null;
  }

  public static ExtensibilityElement getBindingExtension(Binding binding) {
      Collection bindings = new ArrayList();
      filter(bindings, binding.getExtensibilityElements(), HTTPBinding.class);
      filter(bindings, binding.getExtensibilityElements(), SOAPBinding.class);
      if (bindings.size() == 0) {
          return null;
      } else if (bindings.size() > 1) {
          // exception if multiple bindings found
          throw new IllegalArgumentException("Binding "+binding.getQName()+" has multiple binding elements");
      } else {
          // retrieve the single element
          ExtensibilityElement result = (ExtensibilityElement) bindings.iterator().next();
          return result;
      }
  }

  public static ExtensibilityElement getBindingExtension(Port port) {
      Binding binding = port.getBinding();
      if (binding == null) {
          throw new IllegalArgumentException("Binding not found: port "+port.getName()+".");
      }
      return getBindingExtension(binding);
  }

  public static boolean useSOAPBinding(Binding binding) {
      ExtensibilityElement element = getBindingExtension(binding);
      return SOAPBinding.class.isAssignableFrom(element.getClass());
  }

  public static boolean useSOAPBinding(Port port) {
      return useSOAPBinding(port.getBinding());
  }
}
