/*
 * Copyright 2010 Red Hat, Inc
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
package org.jboss.soa.dsp.ws;

import org.jboss.soa.dsp.InvocationAdapter;
import org.jboss.soa.dsp.MessageAdapter;

import javax.xml.namespace.QName;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPMessage;

/**
 * @author Heiko Braun <hbraun@redhat.com>
 */
public final class WSInvocationAdapter implements InvocationAdapter<SOAPMessage>
{
  private final String operationName;
  private final QName serviceName;
  private final String portName;

  private SOAPMessage soapRequestMessage;
  private SOAPMessage soapResponseMessage;

  private final SOAPMessageAdapter soapAdapter;

  public WSInvocationAdapter(String operationName,
      QName serviceName, String portName, SOAPMessageAdapter soapAdapter)
  {
    this.operationName = operationName;
    this.serviceName = serviceName;    
    this.portName = portName;
    this.soapAdapter = soapAdapter;
  }

  public String getOperationName()
  {
    return operationName;
  }

  public QName getServiceName()
  {
    return serviceName;
  }
  
  public String getPortName()
  {
	return portName;
  }

  public void initRequest(javax.wsdl.Operation op, MessageAdapter request)
  {
    if(null==soapRequestMessage)
      throw new IllegalArgumentException("No request SOAPMessage set");
    
    soapAdapter.parseSoapRequest(request, soapRequestMessage, op);
  }
  
  public void createResponse(javax.wsdl.Operation op, MessageAdapter resp)
  {
    try
    {
      // create SOAPEnvelope
      soapResponseMessage = MessageFactory.newInstance().createMessage();
      soapAdapter.createSoapResponse(soapResponseMessage,resp, op);
    }
    catch (SOAPException e)
    {
      throw new RuntimeException("Failed to create SOAP response", e);
    }
  }

  public void createFault(javax.wsdl.Operation op, QName faultName, MessageAdapter fault)
  {
    try
    {
      soapResponseMessage = MessageFactory.newInstance().createMessage();      
      soapAdapter.createSoapFault(soapResponseMessage, fault.getMessage(), faultName, op);
    }
    catch (SOAPException e)
    {
      throw new RuntimeException("Failed to create SOAP fault", e);
    }
  }

  public SOAPMessage getInvocationResult()
  {
    if(null==soapResponseMessage)
      throw new IllegalArgumentException("No response SOAPMessage set");

    return soapResponseMessage; 
  }

  public void setSOAPMessage(SOAPMessage soapMessage)
  {
    this.soapRequestMessage = soapMessage;
  }
}
