/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2007,
 * @author David Fry <dfry@redhat.com>
 * @author Kurt Stam <kurt.stam@jboss.com>
 *
 */
package org.jboss.soa.esb.oracle.aq;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.sql.DataSource;
import javax.sql.XADataSource;

import oracle.jdbc.pool.OracleDataSource;
import oracle.jdbc.xa.client.OracleXADataSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 *
 */
public class AQUtil {
    public static Log log = LogFactory.getLog(AQUtil.class);
    private static final String NORMAL_QUEUE_TYPE = "NORMAL_QUEUE";
    private static final String EXCEPTION_QUEUE_TYPE = "EXCEPTION_QUEUE";
    private static final String QUEUE_RECIPIENTS_STRING = "SINGLE";
    private static final String TOPIC_RECIPIENTS_STRING = "MULTIPLE";
    private static final String GET_QUEUES_SQL = "SELECT allq.OWNER as owner, allq.NAME as name, allq.QUEUE_TYPE as queue_type, allqt.RECIPIENTS as recipients" +
    " from SYS.ALL_QUEUES allq, SYS.ALL_QUEUE_TABLES allqt " +
    " where allq.QUEUE_TABLE = allqt.QUEUE_TABLE " +
    " and upper(allq.OWNER) in (?) and allqt.owner = allq.owner " + 
    " order by name asc";
    private static final String GET_QUEUES_PSQL = "SELECT allq.OWNER as owner, allq.NAME as name, allq.QUEUE_TYPE as queue_type, allqt.RECIPIENTS as recipients" +
    " from SYS.ALL_QUEUES allq, SYS.ALL_QUEUE_TABLES allqt " +
    " where allq.QUEUE_TABLE = allqt.QUEUE_TABLE and upper(allq.OWNER) in (?) order by name asc";


    public static DataSource getSQLDataSource(String serverName,
            String dbInstance, int portNo, String driverName)
            throws SQLException {

        OracleDataSource ds = new OracleDataSource();
        ds.setDatabaseName(dbInstance);
        ds.setDriverType(driverName);
        ds.setServerName(serverName);
        ds.setPortNumber(portNo);
        ds.setNetworkProtocol("tcp");
        return (DataSource) ds;

    }

    public static XADataSource getXADataSource(String serverName,
            String dbInstance, int portNo, String driverName)
            throws SQLException {
        OracleXADataSource ds = new OracleXADataSource();
        ds.setDatabaseName(dbInstance);
        ds.setDriverType(driverName);
        ds.setServerName(serverName);
        ds.setPortNumber(portNo);
        ds.setNetworkProtocol("tcp");
        return (XADataSource) ds;
    }

    public static DataSource getSQLDataSource(String serverName,
            String dbInstance, int portNo, String driverName,
            String username, String password)
            throws SQLException {

        OracleDataSource ds = new OracleDataSource();
        ds.setDatabaseName(dbInstance);
        ds.setDriverType(driverName);
        ds.setServerName(serverName);
        ds.setPortNumber(portNo);
        ds.setNetworkProtocol("tcp");
        ds.setUser(username);
        ds.setPassword(password);
        return ds;
    }

    public static DataSource getSQLDataSource(final String url, final String username, final String password)throws SQLException {

        OracleDataSource ds = new OracleDataSource();
        ds.setURL(url);
        ds.setUser(username);
        ds.setPassword(password);
        return ds;
    }

    public static XADataSource getXASQLDataSource(String serverName,
            String dbInstance, int portNo, String driverName,
            String username, String password)
            throws SQLException {

        OracleXADataSource ds = new OracleXADataSource();
        ds.setDatabaseName(dbInstance);
        ds.setDriverType(driverName);
        ds.setServerName(serverName);
        ds.setPortNumber(portNo);
        ds.setNetworkProtocol("tcp");
        ds.setUser(username);
        ds.setPassword(password);
        return ds;
    }

    public static XADataSource getXASQLDataSource(final String url, final String username, final String password) throws SQLException {

        OracleXADataSource ds = new OracleXADataSource();
        ds.setURL(url);
        ds.setUser(username);
        ds.setPassword(password);
        return ds;
    }

    public static List<DestinationInfo> getDestinationInfoList(DataSource ds, String commaSeperatedQueueOwnerList) {
        ArrayList<DestinationInfo> list = new ArrayList<DestinationInfo>();
        Connection connection = null;
        PreparedStatement statement = null;
        ResultSet rs = null;
        try {
            connection = ds.getConnection();
            rs = null;
            if(commaSeperatedQueueOwnerList!=null) {
                statement = connection.prepareStatement(AQUtil.GET_QUEUES_PSQL);
                statement.setString(1, commaSeperatedQueueOwnerList.toUpperCase());

            }
            else {
                statement = connection.prepareStatement(AQUtil.GET_QUEUES_SQL);
                statement.setString(1, commaSeperatedQueueOwnerList.toUpperCase()); 
            }
            rs = statement.executeQuery();
            while(rs.next()) {
                DestinationInfo info = AQUtil.createDestInfo(rs);
                if(info!=null)list.add(info);
            }
        } catch (SQLException e) {
            log.error("caught sqlexception", e);
        } finally {
            if (rs != null)
            {
        	try {
                    rs.close();
                } catch (SQLException e) {
                    if(log.isDebugEnabled()) log.debug("sqlexception while closing resultset", e);
                }
            }
            if(statement!=null)
                try {
                    statement.close();
                } catch (SQLException e) {
                    if(log.isDebugEnabled()) log.debug("sqlexception while closing prepared statement", e);
                }
            if(connection!=null)
                try {
                    connection.close();
                } catch (SQLException e) {
                    if(log.isDebugEnabled()) log.debug("sqlexception while closing connection", e);
                }
        }
        return list;
    }
    private static DestinationInfo createDestInfo(ResultSet rs) throws SQLException {
        DestinationInfo info = null;
        boolean exceptionDestination = false;
        boolean topic = false;
        String queueType = rs.getString("queue_type");
        String recipients = rs.getString("recipients");
        if(queueType.equalsIgnoreCase(AQUtil.NORMAL_QUEUE_TYPE)) exceptionDestination = false;
        else if(queueType.equalsIgnoreCase(AQUtil.EXCEPTION_QUEUE_TYPE)) exceptionDestination = true;
        else {
            if(log.isDebugEnabled()) log.debug("ignoring unknown queue type:" + queueType);
            return null;
        }
        if(recipients.equalsIgnoreCase(AQUtil.QUEUE_RECIPIENTS_STRING)) topic = false;
        else if(recipients.equalsIgnoreCase(AQUtil.TOPIC_RECIPIENTS_STRING)) topic = true;
        else {
            if(log.isDebugEnabled()) log.debug("ignoring unknown recipients type:" + recipients);
            return null;
        }
        info = new DestinationInfo(rs.getString("owner"), rs.getString("name"), exceptionDestination, topic);
        return info;
    }


}
