package org.jboss.internal.soa.esb.message.format.xml;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.NamedElement;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.stax.TextElement;
import org.jboss.internal.soa.esb.util.stax.URIElement;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.XMLUtil;
/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

public class CallImpl extends ElementContent
{
    /**
     * The call object associated with this type.
     */
    private final Call call ;
    /**
     * Extensions discovered while parsing.
     */
    private List<NamedElement> extensions = new ArrayList<NamedElement>() ;
    
    /**
     * Default constructor for the call.
     */ 
    public CallImpl(final Call call)
    {
        this.call = call ;
    }
    
    /**
     * Construct a context from the input stream.
     * 
     * @param in The input stream.
     * @throws XMLStreamException For errors during parsing.
     */
    public CallImpl(final XMLStreamReader in)
        throws XMLStreamException
    {
        this.call = new Call() ;
        parse(in) ;
        
        if (extensions.size() > 0)
        {
            final EPR origEPR = call.getTo() ;
            final PortReference to = origEPR.getAddr() ;
            for(NamedElement element: extensions)
            {
                final QName name = element.getName() ;
                final TextElement content = (TextElement)element.getElementContent() ;
                to.addExtension(name.getLocalPart(), name.getPrefix(), name.getNamespaceURI(), content.getText()) ;
            }
            
            call.setTo(EPRHelper.getSpecificEPR(origEPR)) ;
        }
    }
    
    /**
     * Return the call object.
     * @return The call object.
     */
    public Call getCall()
    {
        return call ;
    }

    /**
     * Write the child content of the element.
     * @param out The output stream.
     * @throws XMLStreamException For errors during output.
     */
    @Override
    protected void writeChildContent(XMLStreamWriter out)
            throws XMLStreamException
    {
        StreamHelper.writeNamespace(out, XMLUtil.WSA_PREFIX, XMLUtil.WSA_NAMESPACE_URI) ;
        
        if (call.getTo() != null)
        {
            EPRHelper.toXML(out, XMLUtil.QNAME_TO_TAG, call.getTo());
        }

        if (call.getFrom() != null)
        {
            EPRHelper.toXML(out, XMLUtil.QNAME_FROM_TAG, call.getFrom());
        }

        if (call.getReplyTo() != null)
        {
            EPRHelper.toXML(out, XMLUtil.QNAME_REPLY_TO_TAG, call.getReplyTo());
        }

        if (call.getRelatesTo() != null)
        {
            final URIElement actionElement = new URIElement(call.getRelatesTo()) ;
            StreamHelper.writeElement(out, XMLUtil.QNAME_RELATES_TO_TAG, actionElement) ;
        }

        if (call.getFaultTo() != null)
        {
            EPRHelper.toXML(out, XMLUtil.QNAME_FAULT_TO_TAG, call.getFaultTo());
        }

        if (call.getAction() != null)
        {
            final URIElement actionElement = new URIElement(call.getAction()) ;
            StreamHelper.writeElement(out, XMLUtil.QNAME_ACTION_TAG, actionElement) ;
        }

        if (call.getMessageID() != null)
        {
            final URIElement messageIdElement = new URIElement(call.getMessageID()) ;
            StreamHelper.writeElement(out, XMLUtil.QNAME_MESSAGE_IDENTIFIER_TAG, messageIdElement) ;
        }
    }
    
    /**
     * Add the element.
     * @param in The current input stream.
     * @param elementName The qualified element name.
     */
    protected void putElement(final XMLStreamReader in, final QName elementName)
        throws XMLStreamException
    {
        if (XMLUtil.WSA_NAMESPACE_URI.equals(elementName.getNamespaceURI()))
        {
            final String name = elementName.getLocalPart() ;
            if (XMLUtil.TO_TAG.equals(name))
            {
                call.setTo(EPRHelper.fromXML(in));
            }
            else if (XMLUtil.FROM_TAG.equals(name))
            {
                call.setFrom(EPRHelper.fromXML(in));
            }
            else if (XMLUtil.REPLY_TO_TAG.equals(name))
            {
                call.setReplyTo(EPRHelper.fromXML(in));
            }
            else if (XMLUtil.RELATES_TO_TAG.equals(name))
            {
                final URIElement relatesTo = new URIElement(in) ;
                call.setRelatesTo(relatesTo.getValue()) ;
            }
            else if (XMLUtil.FAULT_TO_TAG.equals(name))
            {
                call.setFaultTo(EPRHelper.fromXML(in));
            }
            else if (XMLUtil.ACTION_TAG.equals(name))
            {
                final URIElement action = new URIElement(in) ;
                call.setAction(action.getValue()) ;
            }
            else if (XMLUtil.MESSAGE_IDENTIFIER_TAG.equals(name))
            {
                final URIElement messageIdentifier = new URIElement(in) ;
                call.setMessageID(messageIdentifier.getValue()) ;
            }
            else
            {
                final TextElement content = new TextElement(in) ;
                extensions.add(new NamedElement(elementName, content)) ;
            }
        }
        else
        {
            final TextElement content = new TextElement(in) ;
            extensions.add(new NamedElement(elementName, content)) ;
        }
    }

    public static String toXML(final Call call)
        throws XMLStreamException
    {
        final CallImpl callImpl = new CallImpl(call) ;
        final StringWriter writer = new StringWriter() ;
        final XMLStreamWriter out = XMLHelper.getXMLStreamWriter(writer) ;
        final QName name = org.jboss.internal.soa.esb.message.format.xml.XMLUtil.ESB_QNAME_HEADER ;
        final String origURI = StreamHelper.writeStartElement(out, name) ;
        callImpl.writeContent(out) ;
        StreamHelper.writeEndElement(out, name.getPrefix(), origURI) ;
        out.flush();
        return writer.toString() ;
    }
    
    public static Call fromXML(final String content)
        throws XMLStreamException
    {
        final StringReader reader = new StringReader(content) ;
        final XMLStreamReader in = XMLHelper.getXMLStreamReader(reader) ;
        StreamHelper.checkNextStartTag(in, 
            org.jboss.internal.soa.esb.message.format.xml.XMLUtil.ESB_QNAME_HEADER) ;
        final CallImpl callImpl = new CallImpl(in) ;
        return callImpl.getCall() ;
    }
}