/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.io.Serializable;
import java.util.HashSet;

import javax.jms.BytesMessage;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueBrowser;
import javax.jms.Session;
import javax.jms.StreamMessage;
import javax.jms.TemporaryQueue;
import javax.jms.TemporaryTopic;
import javax.jms.TextMessage;
import javax.jms.Topic;
import javax.jms.TopicSubscriber;

import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool.JmsSessionPool;

/**
 * Wrapper for JMS session class, responsible for tracking resources and the pooling.
 */
public class JmsSession implements Session
{
    /**
     * The connection pool.
     */
    private final JmsConnectionPool connectionPool ;
    /**
     * The session pool.
     */
    private final JmsSessionPool sessionPool ;
    /**
     * The underlying session.
     */
    private final Session session ;
    /**
     * The session delegate.
     */
    private final WrappedSession sessionDelegate ;
    /**
     * The pool instance id.
     */
    private final long id ;
    /**
     * The session acknowledge mode.
     */
    protected final int acknowledgeMode ;
    /**
     * The requested acknowledge mode for this session.
     */
    private final int requestedAcknowledgeMode ;
    /**
     * The flag indicating whether this session is invalid.
     */
    protected volatile boolean invalid ;

    /**
     * Flag indicating whether this session is suspect or not.
     */
    private boolean suspect ;
    
    /**
     * The set of active queue browsers.
     */
    private HashSet<QueueBrowser> queueBrowserSet ;
    /**
     * The set of active message consumers.
     */
    private HashSet<MessageConsumer> messageConsumerSet ;
    /**
     * The set of active message producers.
     */
    private HashSet<MessageProducer> messageProducerSet ;
    
    /**
     * Duplicate the session wrapper.
     * @param connectionPool The connection pool associated with this session.
     * @param sessionPool The session pool associated with this session.
     * @param session The wrapped session delegate.
     * @param id The pool instance id.
     * @param requestedAcknowledgeMode The requested acknowledge mode for this session.
     * @param acknowledgeMode The original acknowledge mode for this session.
     */
    JmsSession(final JmsConnectionPool connectionPool, final JmsSessionPool sessionPool, final Session session, final long id, final int requestedAcknowledgeMode,
            final int acknowledgeMode)
    {
        this(connectionPool, sessionPool, new WrappedSession(connectionPool, session), session, id, requestedAcknowledgeMode, acknowledgeMode) ;
    }

    /**
     * Duplicate the session wrapper.
     * @param connectionPool The connection pool associated with this session.
     * @param sessionPool The session pool associated with this session.
     * @param sessionDelegate The wrapped session delegate.
     * @param session The session.
     * @param id The pool instance id.
     * @param requestedAcknowledgeMode The requested acknowledge mode for this session.
     * @param acknowledgeMode The original acknowledge mode for this session.
     */
    JmsSession(final JmsConnectionPool connectionPool, final JmsSessionPool sessionPool, final WrappedSession sessionDelegate, final Session session,
        final long id, final int requestedAcknowledgeMode, final int acknowledgeMode)
    {
        this.connectionPool = connectionPool ;
        this.sessionPool = sessionPool ;
        this.id = id ;
        this.session = session ;
        this.sessionDelegate = sessionDelegate ;
        sessionDelegate.setJmsSession(this) ;
        this.requestedAcknowledgeMode = requestedAcknowledgeMode ;
        this.acknowledgeMode = acknowledgeMode ;
    }

    /**
     * Create the session wrapper.
     * @param connectionPool The connection pool associated with this session.
     * @param sessionPool The session pool associated with this session.
     * @param session The session delegate.
     * @param id The pool instance id.
     * @param requestedAcknowledgeMode The requested acknowledge mode for this session.
     * @throws JMSException
     */
    JmsSession(final JmsConnectionPool connectionPool, final JmsSessionPool sessionPool, final Session session, final long id, final int requestedAcknowledgeMode)
        throws JMSException
    {
        this(connectionPool, sessionPool, new WrappedSession(connectionPool, session), session, id, requestedAcknowledgeMode) ;
    }

    /**
     * Create the session wrapper.
     * @param connectionPool The connection pool associated with this session.
     * @param sessionPool The session pool associated with this session.
     * @param sessionDelegate The wrapped session delegate.
     * @param session The session.
     * @param id The pool instance id.
     * @param requestedAcknowledgeMode The requested acknowledge mode for this session.
     * @throws JMSException
     */
    JmsSession(final JmsConnectionPool connectionPool, final JmsSessionPool sessionPool, final WrappedSession sessionDelegate, final Session session,
        final long id, final int requestedAcknowledgeMode)
        throws JMSException
    {
        this.connectionPool = connectionPool ;
        this.sessionPool = sessionPool ;
        this.id = id ;
        this.session = session ;
        this.sessionDelegate = sessionDelegate ;
        sessionDelegate.setJmsSession(this) ;
        this.requestedAcknowledgeMode = requestedAcknowledgeMode ;
        acknowledgeMode = session.getAcknowledgeMode() ;
        // Workaround for JBESB-1873
        if ("org.jboss.jms.client.JBossSession".equals(session.getClass().getName()))
        {
            SessionExecutorInit.initExecutor(session) ;
        }
    }

    public long getId()
    {
        return id ;
    }
    
    public void close() throws JMSException
    {
        sessionDelegate.close();
    }

    public void commit() throws JMSException
    {
        setSuspect(true) ;
        sessionDelegate.commit();
        setSuspect(false) ;
    }

    public QueueBrowser createBrowser(Queue arg0, String arg1)
            throws JMSException
    {
        return trackQueueBrowser(sessionDelegate.createBrowser(arg0, arg1));
    }

    public QueueBrowser createBrowser(Queue arg0) throws JMSException
    {
        return trackQueueBrowser(sessionDelegate.createBrowser(arg0));
    }

    public BytesMessage createBytesMessage() throws JMSException
    {
        associate() ;
        return sessionDelegate.createBytesMessage();
    }

    public MessageConsumer createConsumer(Destination arg0, String arg1,
            boolean arg2) throws JMSException
    {
        return trackMessageConsumer(sessionDelegate.createConsumer(arg0, arg1, arg2));
    }

    public MessageConsumer createConsumer(Destination arg0, String arg1)
            throws JMSException
    {
        return trackMessageConsumer(sessionDelegate.createConsumer(arg0, arg1));
    }

    public MessageConsumer createConsumer(Destination arg0) throws JMSException
    {
        return trackMessageConsumer(sessionDelegate.createConsumer(arg0));
    }

    public TopicSubscriber createDurableSubscriber(Topic arg0, String arg1,
            String arg2, boolean arg3) throws JMSException
    {
        return trackTopicSubscriber(sessionDelegate.createDurableSubscriber(arg0, arg1, arg2, arg3));
    }

    public TopicSubscriber createDurableSubscriber(Topic arg0, String arg1)
            throws JMSException
    {
        return trackTopicSubscriber(sessionDelegate.createDurableSubscriber(arg0, arg1));
    }

    public MapMessage createMapMessage() throws JMSException
    {
        associate() ;
        return sessionDelegate.createMapMessage();
    }

    public Message createMessage() throws JMSException
    {
        associate() ;
        return sessionDelegate.createMessage();
    }

    public ObjectMessage createObjectMessage() throws JMSException
    {
        associate() ;
        return sessionDelegate.createObjectMessage();
    }

    public ObjectMessage createObjectMessage(Serializable arg0)
            throws JMSException
    {
        associate() ;
        return sessionDelegate.createObjectMessage(arg0);
    }

    public MessageProducer createProducer(Destination arg0) throws JMSException
    {
        return trackMessageProducer(sessionDelegate.createProducer(arg0));
    }

    public Queue createQueue(String arg0) throws JMSException
    {
        associate() ;
        return sessionDelegate.createQueue(arg0);
    }

    public StreamMessage createStreamMessage() throws JMSException
    {
        associate() ;
        return sessionDelegate.createStreamMessage();
    }

    public TemporaryQueue createTemporaryQueue() throws JMSException
    {
        associate() ;
        return sessionDelegate.createTemporaryQueue();
    }

    public TemporaryTopic createTemporaryTopic() throws JMSException
    {
        associate() ;
        return sessionDelegate.createTemporaryTopic();
    }

    public TextMessage createTextMessage() throws JMSException
    {
        associate() ;
        return sessionDelegate.createTextMessage();
    }

    public TextMessage createTextMessage(String arg0) throws JMSException
    {
        associate() ;
        return sessionDelegate.createTextMessage(arg0);
    }

    public Topic createTopic(String arg0) throws JMSException
    {
        associate() ;
        return sessionDelegate.createTopic(arg0);
    }

    public int getAcknowledgeMode() throws JMSException
    {
        return acknowledgeMode;
    }

    public MessageListener getMessageListener() throws JMSException
    {
        associate() ;
        return sessionDelegate.getMessageListener();
    }

    public boolean getTransacted() throws JMSException
    {
        associate() ;
        return sessionDelegate.getTransacted();
    }

    public void recover() throws JMSException
    {
        associate() ;
        sessionDelegate.recover();
    }

    public void rollback() throws JMSException
    {
        setSuspect(true) ;
        sessionDelegate.rollback();
        setSuspect(false) ;
    }

    public void run()
    {
        sessionDelegate.run();
    }

    public void setMessageListener(MessageListener arg0) throws JMSException
    {
        associate() ;
        sessionDelegate.setMessageListener(arg0);
    }

    public void unsubscribe(String arg0) throws JMSException
    {
        associate() ;
        sessionDelegate.unsubscribe(arg0);
    }

    private synchronized QueueBrowser trackQueueBrowser(QueueBrowser queueBrowser)
        throws JMSException
    {
        associate() ;
        if (queueBrowserSet == null)
        {
            queueBrowserSet = new HashSet<QueueBrowser>() ;
        }
        queueBrowserSet.add(queueBrowser) ;
        return getQueueBrowser(queueBrowser) ;
    }

    private synchronized MessageConsumer trackMessageConsumer(MessageConsumer messageConsumer)
        throws JMSException
    {
        associate() ;
        if (messageConsumerSet == null)
        {
            messageConsumerSet = new HashSet<MessageConsumer>() ;
        }
        messageConsumerSet.add(messageConsumer) ;
        return getMessageConsumer(messageConsumer) ;
    }

    private synchronized TopicSubscriber trackTopicSubscriber(TopicSubscriber topicSubscriber)
        throws JMSException
    {
        associate() ;
        if (messageConsumerSet == null)
        {
            messageConsumerSet = new HashSet<MessageConsumer>() ;
        }
        messageConsumerSet.add(topicSubscriber) ;
        return getTopicSubscriber(topicSubscriber) ;
    }

    private synchronized MessageProducer trackMessageProducer(MessageProducer messageProducer)
        throws JMSException
    {
        associate() ;
        if (messageProducerSet == null)
        {
            messageProducerSet = new HashSet<MessageProducer>() ;
        }
        messageProducerSet.add(messageProducer) ;
        return getMessageProducer(messageProducer) ;
    }

    synchronized void releaseResources()
    {
        if (queueBrowserSet != null)
        {
            for(QueueBrowser queueBrowser: queueBrowserSet)
            {
                try {
                    queueBrowser.close() ;
                } catch (final Exception jmse) {} // ignore
            }
            queueBrowserSet = null ;
        }
        if (messageConsumerSet != null)
        {
            for(MessageConsumer messageConsumer: messageConsumerSet)
            {
                try {
                    messageConsumer.close() ;
                } catch (final Exception jmse) {} // ignore
            }
            messageConsumerSet = null ;
        }
        if (messageProducerSet != null)
        {
            for(MessageProducer messageProducer: messageProducerSet)
            {
                try {
                    messageProducer.close() ;
                } catch (final Exception jmse) {} // ignore
            }
            messageProducerSet = null ;
        }
    }

    protected QueueBrowser getQueueBrowser(QueueBrowser queueBrowser)
    {
        return new WrappedQueueBrowser(connectionPool, this, queueBrowser) ;
    }

    protected MessageConsumer getMessageConsumer(MessageConsumer messageConsumer)
    {
        return new WrappedMessageConsumer(connectionPool, this, messageConsumer) ;
    }

    protected TopicSubscriber getTopicSubscriber(TopicSubscriber topicSubscriber)
    {
        return new WrappedTopicSubscriber(connectionPool, this, topicSubscriber) ;
    }

    protected MessageProducer getMessageProducer(MessageProducer messageProducer)
    {
        return new WrappedMessageProducer(connectionPool, this, messageProducer) ;
    }

    protected void handleCloseSession(final JmsConnectionPool jmsConnectionPool)
    {
        jmsConnectionPool.handleCloseSession(this) ;
    }

    protected void handleReleaseSession(JmsConnectionPool jmsConnectionPool)
    {
        jmsConnectionPool.handleReleaseSession(this) ;
    }
    
    protected void associate()
        throws JMSException
    {
        if (invalid)
        {
            throw new JMSException("Session no longer valid") ;
        }
    }
    
    protected JmsSessionPool getSessionPool()
    {
        return sessionPool ;
    }

    protected Session getSession()
    {
        return session ;
    }
    
    protected void setSuspect(final boolean suspect)
    {
        this.suspect = suspect ;
    }
    
    public boolean isSuspect()
    {
        return suspect ;
    }
    
    public int getRequestedAcknowledgeMode()
    {
        return requestedAcknowledgeMode ;
    }
    
    public boolean isInvalid()
    {
        return invalid ;
    }
    
    protected WrappedSession getSessionDelegate()
    {
        return sessionDelegate ;
    }
    
    public JmsSession duplicateSession()
    {
        invalid = true ;
        return new JmsSession(connectionPool, sessionPool, session, id, requestedAcknowledgeMode, acknowledgeMode) ;
    }
}
