/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.util.concurrent.atomic.AtomicBoolean;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;

/**
 * Wrapped message producer class.
 */
class WrappedMessageProducer implements MessageProducer
{
    /**
     * The connection pool associated with the session.
     */
    private final JmsConnectionPool connectionPool ;
    /**
     * The session associated with the producer.
     */
    private final JmsSession jmsSession ;
    /**
     * The message producer.
     */
    private final MessageProducer messageProducer ;
    /**
     * valid flag for xa associations.
     */
    private final AtomicBoolean wrapperValid ;

    /**
     * Construct the wrapped message producer.
     * @param connectionPool The connection pool associated with the session.
     * @param jmsSession The session associated with the producer.
     * @param messageProducer The message producer.
     */
    public WrappedMessageProducer(final JmsConnectionPool connectionPool,
        final JmsSession jmsSession, final MessageProducer messageProducer)
    {
        this(connectionPool, jmsSession, messageProducer, null) ;
    }

    /**
     * Construct the wrapped message producer.
     * @param connectionPool The connection pool associated with the session.
     * @param jmsSession The session associated with the producer.
     * @param messageProducer The message producer.
     * @param wrapperValid valid flag for xa associations.
     */
    public WrappedMessageProducer(final JmsConnectionPool connectionPool,
        final JmsSession jmsSession, final MessageProducer messageProducer,
        final AtomicBoolean wrapperValid)
    {
        this.connectionPool = connectionPool ;
        this.jmsSession = jmsSession ;
        this.messageProducer = messageProducer ;
        this.wrapperValid = wrapperValid ;
    }

    public void close()
        throws JMSException
    {
        validate() ;
        try
        {
            messageProducer.close() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public int getDeliveryMode()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageProducer.getDeliveryMode() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public Destination getDestination()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageProducer.getDestination() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public boolean getDisableMessageID()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageProducer.getDisableMessageID() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public boolean getDisableMessageTimestamp()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageProducer.getDisableMessageTimestamp() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public int getPriority()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageProducer.getPriority() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public long getTimeToLive()
        throws JMSException
    {
        associate() ;
        try
        {
            return messageProducer.getTimeToLive() ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void send(Destination arg0, Message arg1, int arg2, int arg3,
            long arg4)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.send(arg0, arg1, arg2, arg3, arg4) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void send(Destination arg0, Message arg1)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.send(arg0, arg1) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void send(Message arg0, int arg1, int arg2, long arg3)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.send(arg0, arg1, arg2, arg3) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void send(Message arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.send(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void setDeliveryMode(int arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.setDeliveryMode(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void setDisableMessageID(boolean arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.setDisableMessageID(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void setDisableMessageTimestamp(boolean arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.setDisableMessageTimestamp(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void setPriority(int arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.setPriority(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    public void setTimeToLive(long arg0)
        throws JMSException
    {
        associate() ;
        try
        {
            messageProducer.setTimeToLive(arg0) ;
        }
        catch (final JMSException jmse)
        {
            connectionPool.handleException(jmsSession, jmse) ;
            throw jmse ;
        }
    }

    private void validate()
        throws JMSException
    {
        if ((wrapperValid != null) && !wrapperValid.get())
        {
            throw new JMSException("Instance is no longer valid") ;
        }
    }

    private void associate()
        throws JMSException
    {
        validate() ;
        jmsSession.associate() ;
    }
}
