/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.util.ArrayList;
import java.util.List;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Simple SAX parser creating an identity document for the incoming XML.
 * Any leading and trailing whitespace is ignored in the document as are
 * namespace prefixes.
 *
 * @author Kevin Conner
 */
public class IdentitySAXHandler extends DefaultHandler
{
    /**
     * The root element.
     */
    private Element rootElement ;
    /**
     * The current element.
     */
    private Element currentElement ;
    /**
     * The stack of working elements.
     */
    private List<Element> stack = new ArrayList<Element>() ;
    /**
     * The current text value.
     */
    private StringBuilder currentText = new StringBuilder() ;

    @Override
    public void startElement(final String uri, final String localName,
        final String name, final Attributes attributes)
        throws SAXException
    {
        checkText() ;
        
        final Element element = new Element(uri, localName, attributes) ;
        if (rootElement == null)
        {
            rootElement = element ;
        }
        
        if (currentElement != null)
        {
            currentElement.addChild(element) ;
            stack.add(currentElement) ;
        }
        currentElement = element ;
    }
    
    @Override
    public void endElement(final String uri, final String localName,
        final String name)
        throws SAXException
    {
        checkText() ;
        
        final int lastIndex = (stack.size() - 1) ;
        if (lastIndex < 0)
        {
            currentElement = null ;
        }
        else
        {
            currentElement = stack.remove(lastIndex) ;
        }
    }
    
    @Override
    public void characters(final char[] ch, final int start, final int length)
        throws SAXException
    {
        currentText.append(ch, start, length) ;
    }
    
    private void checkText()
    {
        final int textLength = currentText.length() ;
        if (textLength > 0)
        {
            int start = 0 ;
            while((start < textLength) && isXMLWhitespace(currentText.charAt(start)))
            {
                start++ ;
            }
            
            int end = textLength-1 ;
            while((end >= start) && isXMLWhitespace(currentText.charAt(end)))
            {
                end-- ;
            }
            
            if (start <= end)
            {
                currentElement.addChild(new Text(currentText.substring(start, end+1))) ;
            }
            currentText.setLength(0) ;
            currentText.trimToSize() ;
        }
    }
    
    private boolean isXMLWhitespace(final char ch)
    {
        return ((ch == ' ') || (ch == '\t') || (ch == '\r') || (ch == '\n')) ;
    }
    
    public Element getRootElement()
    {
        return rootElement ;
    }
}
