/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.ConfigurationException;
import org.apache.log4j.Logger;

import java.io.*;
import java.net.URI;
import java.net.URISyntaxException;

/**
 * Stream handling utilities.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class StreamUtils {

    private static Logger logger = Logger.getLogger(StreamUtils.class);

    /**
	 * Read the supplied InputStream and return as an array of bytes.
	 * @param stream The stream to read.
	 * @return The stream contents in an array of bytes.
	 */
	public static byte[] readStream(InputStream stream) {
		if(stream == null) {
			throw new IllegalArgumentException("null 'stream' arg passed in method call.");
		}
		
		ByteArrayOutputStream outBuffer = new ByteArrayOutputStream();
		byte[] buffer = new byte[256];
		int readCount = 0;
		
		try {
			while((readCount = stream.read(buffer)) != -1) {
				outBuffer.write(buffer, 0, readCount);
			}
		} catch (IOException e) {
			throw new IllegalStateException("Error reading stream.", e);
		}		
		
		return outBuffer.toByteArray();
	}
	
       /**
         * Read the supplied Reader and return as a string.
         * @param reader The reader to read.
         * @return The reader contents as a string.
         */
        public static String readReader(Reader reader) {
                if(reader == null) {
                        throw new IllegalArgumentException("null 'reader' arg passed in method call.");
                }
                
                StringWriter outBuffer = new StringWriter();
                char[] buffer = new char[256];
                int readCount = 0;
                
                try {
                        while((readCount = reader.read(buffer)) != -1) {
                                outBuffer.write(buffer, 0, readCount);
                        }
                } catch (IOException e) {
                        throw new IllegalStateException("Error reading reader.", e);
                }               
                
                return outBuffer.toString();
        }

    public static String readStreamString(InputStream stream, String charset) throws UnsupportedEncodingException {
        return new String(StreamUtils.readStream(stream), charset);
    }

    public static String getResourceAsString(String resource, String charset) throws ConfigurationException, UnsupportedEncodingException {
        InputStream stream = getResource(resource);

        try {
            return readStreamString(stream, charset);
        } finally {
            try {
                stream.close();
            } catch (IOException e) {
                logger.error("Error closing resource stream '" + resource + "'.", e);
            }
        }
    }
    
    public static byte[] getResourceAsBytes(String resource) throws ConfigurationException {
        InputStream stream = getResource(resource);

        try {
            return readStream(stream);
        } finally {
            try {
                stream.close();
            } catch (IOException e) {
                logger.error("Error closing resource stream '" + resource + "'.", e);
            }
        }
    }

    /**
     * Load a stream resource.
     * @param resource Resource filesystem/classpath/URL location.
     * @return The resource stream.
     * @throws ConfigurationException Failed to load the resource.
     */
    public static InputStream getResource(String resource) throws ConfigurationException {
    	return getResource(resource, StreamUtils.class);
    }
    
    /**
     * Load a stream resource.
     * @param resource Resource filesystem/classpath/URL location.
     * @return The resource stream.
     * @param caller The class of the caller.
     * @throws ConfigurationException Failed to load the resource.
     */
    public static InputStream getResource(String resource, Class<?> caller) throws ConfigurationException {
        try {
            // Try it as a File resource...
            File file = new File(resource);

            if(file.exists() && !file.isDirectory()) {
                return new FileInputStream(file);
            }

            // Try it as a classpath resource ...
            InputStream stream = ClassUtil.getResourceAsStream(resource, caller);
            if(stream != null) {
                return stream;
            }

            // Try it as a URI resource...
            try {
                URI fileURI = new URI(resource);
                if(fileURI.isAbsolute()) {
                    return fileURI.toURL().openStream();
                }
            } catch (URISyntaxException e) {
                throw new ConfigurationException("Failed to load resource '" + resource + "'.");
            }
        } catch (IOException e) {
            throw new ConfigurationException("Failed to load resource '" + resource + "'.", e);
        }

        throw new ConfigurationException("Failed to locate resource '" + resource + "'.");
    }
}
