/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. 
 * See the copyright.txt in the distribution for a full listing 
 * of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License, v. 2.0.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License,
 * v. 2.0 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
package org.jboss.internal.soa.esb.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.concurrent.atomic.AtomicReference;

import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.stream.events.XMLEvent;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.stax.ParsingSupport;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.wstx.ESBDOMWrappingReader;
import org.jboss.internal.soa.esb.util.wstx.ESBDOMWrappingWriter;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.util.StringPropertyReplacer;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.ls.LSResourceResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import com.ctc.wstx.stax.WstxInputFactory;
import com.ctc.wstx.stax.WstxOutputFactory;

/**
 * Helper class for manipulating XML documents.
 * 
 * @author <a href='mailto:kevin.conner@jboss.com'>Kevin Conner</a>
 */
public class XMLHelper
{
    private static Logger log = Logger.getLogger(XMLHelper.class);
    
    /**
     * The XML input factory.
     */
    private static final XMLInputFactory XML_INPUT_FACTORY  ;
    /**
     * The XML output factory.
     */
    private static final XMLOutputFactory XML_OUTPUT_FACTORY ;
    /**
     * The Document builder factory.
     */
    private static final DocumentBuilderFactory DOCUMENT_BUILDER_FACTORY ;
    /**
     * The Document builder for document creation (not parsing).
     */
    private static final AtomicReference<DocumentBuilder> DOCUMENT_BUILDER = new AtomicReference<DocumentBuilder>();
    /**
     * The event writer creator for DOM documents.
     */
    private static final EventWriterCreator EVENT_WRITER_CREATOR ;
    /**
     * The event reader creator for DOM nodes.
     */
    private static final EventReaderCreator EVENT_READER_CREATOR ;
    
    /**
     * Get the XML stream reader.
     * @param reader The input reader.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final Reader reader)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(reader) ;
    }

    /**
     * Get the XML stream reader.
     * @param is The input stream.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final InputStream is)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(is) ;
    }

    /**
     * Get the XML stream reader.
     * @param is The input stream.
     * @param encoding The input stream encoding.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final InputStream is, final String encoding)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(is, encoding) ;
    }

    /**
     * Get the XML stream reader.
     * @param source The source.
     * @return The XML stream reader.
     * @throws XMLStreamException For errors obtaining an XML stream reader.
     */
    public static XMLStreamReader getXMLStreamReader(final Source source)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLStreamReader(source) ;
    }

    /**
     * Get the XML event reader.
     * @param reader The input reader.
     * @return The XML event reader.
     * @throws XMLStreamException For errors obtaining an XML event reader.
     */
    public static XMLEventReader getXMLEventReader(final Reader reader)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLEventReader(reader) ;
    }

    /**
     * Get the XML event reader.
     * @param is The input stream.
     * @return The XML event reader.
     * @throws XMLStreamException For errors obtaining an XML event reader.
     */
    public static XMLEventReader getXMLEventReader(final InputStream is)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLEventReader(is) ;
    }

    /**
     * Get the XML event reader.
     * @param is The input stream.
     * @param encoding The input stream encoding.
     * @return The XML event reader.
     * @throws XMLStreamException For errors obtaining an XML event reader.
     */
    public static XMLEventReader getXMLEventReader(final InputStream is, final String encoding)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLEventReader(is, encoding) ;
    }

    /**
     * Get the XML event reader.
     * @param source The source.
     * @return The XML event reader.
     * @throws XMLStreamException For errors obtaining an XML event reader.
     */
    public static XMLEventReader getXMLEventReader(final Source source)
        throws XMLStreamException
    {
        return XML_INPUT_FACTORY.createXMLEventReader(source) ;
    }

    /**
     * Get the XML stream writer.
     * @param writer The output writer.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final Writer writer)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(writer) ;
    }

    /**
     * Get the XML stream writer.
     * @param os The output stream.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final OutputStream os)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(os) ;
    }

    /**
     * Get the XML stream writer.
     * @param os The output stream.
     * @param encoding The output stream encoding.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final OutputStream os, final String encoding)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(os, encoding) ;
    }

    /**
     * Get the XML stream writer.
     * @param result The output result.
     * @return The XML stream writer.
     * @throws XMLStreamException For errors obtaining an XML stream writer.
     */
    public static XMLStreamWriter getXMLStreamWriter(final Result result)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLStreamWriter(result) ;
    }

    /**
     * Get the XML event writer.
     * @param writer The output writer.
     * @return The XML event writer.
     * @throws XMLStreamException For errors obtaining an XML event writer.
     */
    public static XMLEventWriter getXMLEventWriter(final Writer writer)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLEventWriter(writer) ;
    }

    /**
     * Get the XML event writer.
     * @param os The output stream.
     * @return The XML event writer.
     * @throws XMLStreamException For errors obtaining an XML event writer.
     */
    public static XMLEventWriter getXMLEventWriter(final OutputStream os)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLEventWriter(os) ;
    }

    /**
     * Get the XML event writer.
     * @param os The output stream.
     * @param encoding The output stream encoding.
     * @return The XML event writer.
     * @throws XMLStreamException For errors obtaining an XML event writer.
     */
    public static XMLEventWriter getXMLEventWriter(final OutputStream os, final String encoding)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLEventWriter(os, encoding) ;
    }

    /**
     * Get the XML event writer.
     * @param result The output result.
     * @return The XML event writer.
     * @throws XMLStreamException For errors obtaining an XML event writer.
     */
    public static XMLEventWriter getXMLEventWriter(final Result result)
        throws XMLStreamException
    {
        return XML_OUTPUT_FACTORY.createXMLEventWriter(result) ;
    }
    
    /**
     * Copy an XML event stream.
     * @param reader The event reader.
     * @param writer The event writer.
     * @throws XMLStreamException For errors writing to the XML event writer.
     */
    public static void copyXMLEventStream(final XMLEventReader reader, final XMLEventWriter writer)
        throws XMLStreamException
    {
        copyXMLEventStream(reader, writer, false) ;
    }
    
    /**
     * Copy an XML event stream.
     * @param reader The event reader.
     * @param writer The event writer.
     * @param omitDoc if true, ignore start/end document events.
     * @throws XMLStreamException For errors writing to the XML event writer.
     */
    public static void copyXMLEventStream(final XMLEventReader reader, final XMLEventWriter writer, final boolean omitDoc)
        throws XMLStreamException
    {
        if (omitDoc)
        {
            while(reader.hasNext())
            {
                final XMLEvent event = reader.nextEvent() ;
                final int type = event.getEventType() ;
                if ((type != XMLStreamConstants.START_DOCUMENT) &&
                    (type != XMLStreamConstants.END_DOCUMENT))
                {
                    writer.add(event) ;
                }
            }
        }
        else
        {
            writer.add(reader) ;
        }
        writer.flush() ;
    }
    
    /**
     * Replace system property values within the attribute values/text elements.
     * @param streamReader The XML stream reader.
     * @param streamWriter The XMl stream writer.
     * @throws XMLStreamException For errors during parsing.
     */
    public static void replaceSystemProperties(final XMLStreamReader streamReader,
        final XMLStreamWriter streamWriter)
        throws XMLStreamException
    {
        streamWriter.writeStartDocument() ;
        
        StreamHelper.skipToStartElement(streamReader) ;
        final QName elementName = streamReader.getName() ;
        final String uri = StreamHelper.writeStartElement(streamWriter, elementName) ;
        
        new SystemPropertyReplacementParser(streamReader, streamWriter) ;
        
        StreamHelper.writeEndElement(streamWriter, elementName.getPrefix(), uri) ;
        
        streamWriter.writeEndDocument() ;
        streamWriter.flush() ;
    }
    
    /**
     * Get the schema for the specified resource.
     * @param resource The schema resource to parse.
     * @return The resource schema for validation. 
     * @throws SAXException For errors during parsing.
     */
    public static Schema getSchema(final String resource) throws SAXException
    {
        return getSchema(resource, XMLConstants.W3C_XML_SCHEMA_NS_URI) ;
    }
    
    /**
     * Get the schema for the specified resource.
     * @param resource The schema resource to parse.
     * @param schemaLanguage The schema language.
     * @return The resource schema for validation. 
     * @throws SAXException For errors during parsing.
     */
    public static Schema getSchema(final String resource, final String schemaLanguage) throws SAXException
    {
        SchemaResolver schemaResolver;
        try
        {
            URI schemaUri = getResourceUri(resource, XMLHelper.class) ;
            final boolean debugEnabled = log.isDebugEnabled() ;
            if (debugEnabled)
            {
                log.debug("schemaUri : " + schemaUri);
            }
            schemaResolver = new SchemaResolver(schemaUri);
            
            URL schemaUrl;
            if (schemaUri.getScheme().equals("classpath"))
            {
                schemaUrl = ClassUtil.getResource(schemaUri.getPath(), XMLHelper.class) ;
            }
            else
            {
                schemaUrl = schemaUri.toURL();
            }
            if (debugEnabled)
            {
                log.debug("schemaUrl : " + schemaUrl);
            }
            
            return getSchema(schemaUrl, schemaResolver, schemaLanguage);
        } 
        catch (final URISyntaxException e)
        {
            throw new SAXException("URISyntaxException while trying to locate '" + resource + "'");
        } 
        catch (final MalformedURLException e)
        {
            throw new SAXException("MalformedURLException while trying to located '" + resource + "'");
        } 
        catch (final IOException e)
        {
            throw new SAXException("IOException while trying to locate '" + resource + "'");
        }
    }
    
    /**
     * Get the schema for the specified resource and use the specified resolver to locate 
     * external resources, for example import schemas in the xsd.
     * 
     * @param resource  The schema resource to parse.
     * @param resolver  The {@link LSResourceResolver} for locating external resources.
     * @return Schema   The resource schema for validation. 
     * @throws SAXException
     * @throws IOException 
     */
    public static Schema getSchema(final URL resource, final LSResourceResolver resolver) throws SAXException, IOException
    {
        return getSchema(resource, resolver, XMLConstants.W3C_XML_SCHEMA_NS_URI) ;
    }
    
    /**
     * Get the schema for the specified resource and use the specified resolver to locate 
     * external resources, for example import schemas in the xsd.
     * 
     * @param resource  The schema resource to parse.
     * @param resolver  The {@link LSResourceResolver} for locating external resources.
     * @param schemaLanguage The schema language.
     * @return Schema   The resource schema for validation. 
     * @throws SAXException
     * @throws IOException 
     */
    public static Schema getSchema(final URL resource, final LSResourceResolver resolver, final String schemaLanguage) throws SAXException, IOException
    {
        final SchemaFactory schemaFactory = newSchemaFactory(schemaLanguage);
        schemaFactory.setResourceResolver(resolver);
        
        final InputStream resourceIS = resource.openStream();
        return schemaFactory.newSchema(new StreamSource(resourceIS)) ;
    }
    
    public static URI getResourceUri(final String resourceName, final Class<?> caller) throws URISyntaxException, MalformedURLException
    {
        File file = new File(resourceName);
        if (file.exists() && !file.isDirectory())
        {
            return file.toURI();
        }
        
        String resource = resourceName;
        if (resourceName.startsWith("/"))
        {
            resource = resourceName.substring(1) ;
        }
        
        final ClassLoader threadClassLoader = Thread.currentThread().getContextClassLoader() ;
        if (threadClassLoader != null)
        {
            final URL url = threadClassLoader.getResource(resource) ;
            if (url != null)
            {
                return URI.create("classpath://" + resourceName);
            }
        }
        
        final ClassLoader classLoader = caller.getClassLoader() ;
        if (classLoader != null)
        {
            final URL url = classLoader.getResource(resource) ;
            if (url != null)
            {
                return URI.create("classpath://" + resourceName);
            }
        }
        
        URL systemResource = ClassLoader.getSystemResource(resource);
        if (systemResource != null)
        {
            return systemResource.toURI();
        }
        
        return new URI(resourceName);
    }

    
    /**
     * Get the schema for the specified resources.
     * @param resources The schema resources to parse.
     * @return The resource schema for validation. 
     * @throws SAXException For errors during parsing.
     */
    public static Schema getSchema(final String[] resources)
        throws SAXException
    {
        final int numResources = (resources == null ? 0 : resources.length) ;
        final Source[] sources = new Source[numResources] ;
        for(int count = 0 ; count < numResources ; count++)
        {
            final InputStream resourceIS = ClassUtil.getResourceAsStream(resources[count], XMLHelper.class) ;
            sources[count] = new StreamSource(resourceIS) ;
        }
        return newSchemaFactory(XMLConstants.W3C_XML_SCHEMA_NS_URI).newSchema(sources) ;
    }
    
    /**
     * Validate the specified xml against the schema.
     * @param schema The resource schema for validation.
     * @param xml The XML to validate.
     * @return true if valid, false otherwise.
     */
    public static boolean validate(final Schema schema, final String xml)
    {
        final Validator validator = schema.newValidator() ;
        try
        {
            validator.validate(new StreamSource(new StringReader(xml))) ;
            return true ;
        }
        catch (final IOException ioe) 
        {
            log.debug(ioe.getMessage(), ioe);
        } 
        catch (final SAXException saxe)  
        {
            log.debug(saxe.getMessage(), saxe);
        } 
        
        return false ;
    }

    /**
     * Compare the specified contents as XML.
     * @param content1 The first content.
     * @param content2 The second content.
     * @return true if equals, false otherwise.
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    public static boolean compareXMLContent(final InputStream content1, final InputStream content2)
        throws ParserConfigurationException, SAXException, IOException
    {
        return compareXMLContent(new InputSource(content1), new InputSource(content2)) ;
    }
    
    /**
     * Compare the specified contents as XML.
     * @param content1 The first content.
     * @param content2 The second content.
     * @return true if equals, false otherwise.
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    public static boolean compareXMLContent(final String content1, final String content2)
        throws ParserConfigurationException, SAXException, IOException
    {
        return compareXMLContent(new StringReader(content1), new StringReader(content2)) ;
    }
    
    /**
     * Compare the specified contents as XML.
     * @param content1 The first content.
     * @param content2 The second content.
     * @return true if equals, false otherwise.
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    public static boolean compareXMLContent(final Reader content1, final Reader content2)
        throws ParserConfigurationException, SAXException, IOException
    {
        return compareXMLContent(new InputSource(content1), new InputSource(content2)) ;
    }
    
    /**
     * Compare the specified contents as XML.
     * @param content1 The first content.
     * @param content2 The second content.
     * @return true if equals, false otherwise.
     * @throws ParserConfigurationException
     * @throws SAXException
     * @throws IOException
     */
    public static boolean compareXMLContent(final InputSource content1, final InputSource content2)
        throws ParserConfigurationException, SAXException, IOException
    {
        final SAXParserFactory parserFactory = SAXParserFactory.newInstance() ;
        parserFactory.setNamespaceAware(true) ;

        final SAXParser parser = parserFactory.newSAXParser() ;

        final IdentitySAXHandler handler1 = new IdentitySAXHandler() ;
        parser.parse(content1, handler1) ;

        final IdentitySAXHandler handler2 = new IdentitySAXHandler() ;
        parser.parse(content2, handler2) ;

        return (handler1.getRootElement().equals(handler2.getRootElement())) ;
    }

    private static SchemaFactory newSchemaFactory(final String schemaLanguage)
    {
        return SchemaFactory.newInstance(schemaLanguage);
    }

    /**
     * Create a document from the specified reader.
     * @param reader The XMLEvent reader.
     * @return The Document.
     * @throws ParserConfigurationException For errors creating the document.
     * @throws XMLStreamException For errors reading the event reader.
     */
    public static Document createDocument(XMLEventReader reader)
        throws ParserConfigurationException, XMLStreamException
    {
        final Document doc = getNewDocument() ;
        final XMLEventWriter writer = EVENT_WRITER_CREATOR.createXMLEventWriter(doc);
        XMLHelper.copyXMLEventStream(reader, writer) ;
        return doc;
    }
    
    /**
     * Read from a DOM node, output to a writer.
     * @param node The DOM node.
     * @param writer The specified writer.
     * @param omitDoc if true, ignore start/end document events.
     */
    public static void readDomNode(final Node node, final XMLEventWriter writer, final boolean omitDoc)
        throws XMLStreamException
    {
        final XMLEventReader reader = EVENT_READER_CREATOR.createXMLEventReader(node);
        XMLHelper.copyXMLEventStream(reader, writer, omitDoc) ;
    }
    
    /**
     * Create a new document.
     * @return the new document
     * @throws ParserConfigurationException for errors during creation
     */
    private static Document getNewDocument()
        throws ParserConfigurationException
    {
        final DocumentBuilder builder = getCreationDocumentBuilder() ;
        synchronized(builder)
        {
            // synchronized as it is not guaranteed to be thread safe
            return builder.newDocument() ;
        }
    }
    
    /**
     * Get the document builder for creation
     * @return The document builder
     * @throws ParserConfigurationException for errors during creation
     */
    private static DocumentBuilder getCreationDocumentBuilder()
        throws ParserConfigurationException
    {
        final DocumentBuilder current = DOCUMENT_BUILDER.get() ;
        if (current != null)
        {
            return current ;
        }
        final DocumentBuilder newBuilder = DOCUMENT_BUILDER_FACTORY.newDocumentBuilder() ;
        if (DOCUMENT_BUILDER.compareAndSet(null, newBuilder))
        {
            return newBuilder ;
        }
        else
        {
            return DOCUMENT_BUILDER.get() ;
        }
    }

    static
    {
        final XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance() ;
        xmlInputFactory.setProperty(XMLInputFactory.IS_COALESCING, Boolean.TRUE) ;
        XML_INPUT_FACTORY = xmlInputFactory ;

        if ("com.ctc.wstx.stax.WstxInputFactory".equals(XML_INPUT_FACTORY.getClass().getName()))
        {
            EVENT_READER_CREATOR = new WstxEventReaderCreator(XML_INPUT_FACTORY) ;
        }
        else
        {
            EVENT_READER_CREATOR = new DefaultEventReaderCreator() ;
        }

        XML_OUTPUT_FACTORY = XMLOutputFactory.newInstance() ;
        
        if ("com.ctc.wstx.stax.WstxOutputFactory".equals(XML_OUTPUT_FACTORY.getClass().getName()))
        {
            EVENT_WRITER_CREATOR = new WstxEventWriterCreator(XML_OUTPUT_FACTORY) ;
        }
        else
        {
            EVENT_WRITER_CREATOR = new DefaultEventWriterCreator() ;
        }

        final DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
        docBuilderFactory.setNamespaceAware(true);
        DOCUMENT_BUILDER_FACTORY = docBuilderFactory ;
        
    }
    
    /**
     * The parser class used to perform system property replacement.
     * @author kevin
     */
    private static final class SystemPropertyReplacementParser extends ParsingSupport
    {
        /**
         * The output writer.
         */
        private final XMLStreamWriter out ;
        
        /**
         * Construct the parser.
         * @param in The XML input stream.
         * @param out The XML output stream.
         * @throws XMLStreamException For errors during parsing.
         */
        SystemPropertyReplacementParser(final XMLStreamReader in, final XMLStreamWriter out)
            throws XMLStreamException
        {
            this.out = out ;
            parse(in) ;
        }
        
        /**
         * Set the text value of this element.
         * @param in The current input stream.
         * @param value The text value of this element.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putValue(final XMLStreamReader in, final String value)
            throws XMLStreamException
        {
            out.writeCharacters(StringPropertyReplacer.replaceProperties(value)) ;
        }
        
        /**
         * Add the attribute value.
         * @param in The current input stream.
         * @param attributeName The qualified attribute name.
         * @param attributeValue The qualified attribute value.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putAttribute(final XMLStreamReader in,
            final QName attributeName, final String attributeValue)
            throws XMLStreamException
        {
            StreamHelper.writeAttribute(out, attributeName, StringPropertyReplacer.replaceProperties(attributeValue)) ;
        }
        
        /**
         * Add the element.
         * @param in The current input stream.
         * @param elementName The qualified element name.
         * @throws XMLStreamException For errors during parsing.
         */
        protected void putElement(final XMLStreamReader in,
            final QName elementName)
            throws XMLStreamException
        {
            final String uri = StreamHelper.writeStartElement(out, elementName) ;
            new SystemPropertyReplacementParser(in, out) ;
            StreamHelper.writeEndElement(out, elementName.getPrefix(), uri) ;
        }
    }
    
    /**
     * Interface for the event writer creator.
     * @author kevin
     */
    private interface EventWriterCreator
    {
        /**
         * Create the event writer.
         * @param doc The associated document.
         * @return The XML event writer.
         * @throws XMLStreamException for errors constructing the writer.
         */
        public XMLEventWriter createXMLEventWriter(final Document doc)
            throws XMLStreamException ;
    }
    
    /**
     * Interface for the event reader creator.
     * @author kevin
     */
    private interface EventReaderCreator
    {
        /**
         * Create the event reader.
         * @param node The associated node.
         * @return The XML event reader.
         * @throws XMLStreamException for errors constructing the reader.
         */
        public XMLEventReader createXMLEventReader(final Node node)
            throws XMLStreamException ;
    }
    
    /**
     * The default event writer creator
     * @author kevin
     */
    private static final class DefaultEventWriterCreator implements EventWriterCreator
    {
        /**
         * Create the event writer.
         * @param doc The associated document.
         * @return The XML event writer.
         * @throws XMLStreamException for errors constructing the writer.
         */
        public XMLEventWriter createXMLEventWriter(final Document doc)
            throws XMLStreamException
        {
            return getXMLEventWriter(new DOMResult(doc)) ;
        }
    }
    /**
     * The wstx event writer creator
     * @author kevin
     */
    private static final class WstxEventWriterCreator implements EventWriterCreator
    {
        private final WstxOutputFactory outputFactory ;
        
        /**
         * Construct the 
         * @param xmlOutputFactory
         */
        private WstxEventWriterCreator(final XMLOutputFactory xmlOutputFactory)
        {
            outputFactory = (WstxOutputFactory)xmlOutputFactory ;
        }
        
        /**
         * Create the event writer.
         * @param doc The associated document.
         * @return The XML event writer.
         * @throws XMLStreamException for errors constructing the writer.
         */
        public XMLEventWriter createXMLEventWriter(final Document doc)
            throws XMLStreamException
        {
            final XMLStreamWriter wstxWriter = ESBDOMWrappingWriter.createFrom(outputFactory.getConfig(), new DOMResult(doc)) ;
            return outputFactory.createXMLEventWriter(wstxWriter) ;
        }
    }
    
    /**
     * The default event reader creator
     * @author kevin
     */
    private static final class DefaultEventReaderCreator implements EventReaderCreator
    {
        /**
         * Create the event reader.
         * @param node The associated node.
         * @return The XML event reader.
         * @throws XMLStreamException for errors constructing the reader.
         */
        public XMLEventReader createXMLEventReader(final Node node)
            throws XMLStreamException
        {
            return getXMLEventReader(new DOMSource(node)) ;
        }
    }
    /**
     * The wstx event reader creator
     * @author kevin
     */
    private static final class WstxEventReaderCreator implements EventReaderCreator
    {
        private final WstxInputFactory inputFactory ;
        
        /**
         * Construct the 
         * @param xmlOutputFactory
         */
        private WstxEventReaderCreator(final XMLInputFactory xmlInputFactory)
        {
            inputFactory = (WstxInputFactory)xmlInputFactory ;
        }
        
        /**
         * Create the event reader.
         * @param node The associated node.
         * @return The XML event reader.
         * @throws XMLStreamException for errors constructing the reader.
         */
        public XMLEventReader createXMLEventReader(final Node node)
            throws XMLStreamException
        {
            final XMLStreamReader wstxReader = ESBDOMWrappingReader.createFrom(inputFactory.getConfig(), new DOMSource(node)) ;
            return inputFactory.createXMLEventReader(wstxReader) ;
        }
    }
}
