/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.message.Message;

/**
 * Action that can be subclassed to add support for logging. This action is indented
 * for example logging and not production grad logging.
 * <p/>
 * Exemple configuration:
 * <pre>
 *&lt;action name="log" class="org.jboss.soa.esb.actions.LogAction"&gt;
 *	&lt;property name="logLevel" value="info"/&gt;
 *	&lt;property name="logger" value="class name"/&gt;
 *&lt;/action&gt;
 *Options properties
 *	&lt;property name="message" value="Body: "/&gt;
 * </pre>
 * 
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>			
 * @since 4.3
 *
 */
public class LogAction implements BeanConfiguredAction
{
	private String logLevel;
	private String logger;
	private Level level;
	private String message;
	private boolean logBody;
	
	public LogAction()
	{
		logger = getClass().getName();
    	level = Level.DEBUG;
	}
	
	public Message process( final Message msg )
	{
		Object object = msg.getBody().get();
		StringBuilder sb = new StringBuilder();
		if ( message != null )
		{
			sb.append( message );
		}
		if ( logBody )
		{
			sb.append( object );
		}
		log( sb.toString() );
		
		return msg;
	}
	
	protected void log( final String messageToLog )
	{
    	Logger log = Logger.getLogger( logger );
    	log.log( level, messageToLog );
	}
	
	public Level getLevel()
	{
		return level;
	}
	
	public String getLogLevel()
	{
		return logLevel;
	}

	public void setLogLevel( final String logLevel )
	{
    	this.logLevel = logLevel ;
    	level = Level.toLevel( logLevel, Level.DEBUG );
	}
	
	public void setLogger( final String logger )
	{
		if ( logger != null )
    		this.logger = logger;
	}

	public String getLogger()
	{
		return logger;
	}

	public void setMessage( String string )
	{
		message = string;
	}

	public void setLogBody( boolean logBody )
	{
		this.logBody = logBody;
	}

}
