/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.converters.xstream.conf;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.util.ClassUtil;
import com.thoughtworks.xstream.converters.Converter;

/**
 * XStream configuration class that holds information about converters.
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class ConverterConf
{
    private Converter converter;
    private int priority = -1;

    public ConverterConf(final String converterClassName, final String priority)
    {
        AssertArgument.isNotNullAndNotEmpty(converterClassName, "converterClassName");
        this.converter = initialize(converterClassName);
        if (priority != null)
        {
            this.priority = Integer.parseInt(priority);
        }
    }

    public Converter getConverter()
    {
        return converter;
    }

    public int getPriority()
    {
        return priority;
    }

    public String toString()
    {
        return "ConverterConf [converter=" + converter + ", priority=" + priority + "]";
    }
    
    private Converter initialize(final String className)
    {
        try
        {
            Class<?> converterClass = ClassUtil.forName(className, getClass());
            Object newInstance = converterClass.newInstance();
            if (newInstance instanceof Converter)
            {
                return (Converter) newInstance;
            }
            throw new IllegalStateException("Converter '" + className + " does not implement '" + Converter.class.getName() + "'");
        } 
        catch (final ClassNotFoundException e)
        {
            throw new IllegalArgumentException("Could not load class '" + className + "'", e);
        } 
        catch (final InstantiationException e)
        {
            throw new IllegalArgumentException("Could not load class '" + className + "'", e);
        } 
        catch (final IllegalAccessException e)
        {
            throw new IllegalArgumentException("Could not load class '" + className + "'", e);
        }
        
    }

}
