/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.monitoring;

import java.util.ArrayList;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.scripting.ScriptingAction;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;

import bsh.EvalError;
import bsh.Interpreter;

/**
 * Action that monitors messages and has the ability to store information found at a particular MVEL pattern.
 * Uses beanshell evaluation to decide whether to store the message information.
 * 
 * @author tcunning
 * @since Version 4.4
 */
public class MVELMonitoringAction extends AbstractActionPipelineProcessor {

	public static final String MONITOR_VALUE = "monitor-value";
	public static final String MVEL_PATTERN = "mvel-pattern";
	public static final String PATTERN_NAME = "pattern-name";
	public static final String PATTERN_CONDITIONAL = "pattern-conditional";
	
	private static final String BEANSHELL_VARIABLE = "beanshellVar";
	
	private MVELMonitor mvelMonitor;
	
	ArrayList <MonitoringPatternBean> list = null;
	
	private static Logger logger = Logger.getLogger(ScriptingAction.class);
	protected ConfigTree _config;
	
	/**
	 * Constructor that forms 
	 * @param config ConfigTree
	 */
	public MVELMonitoringAction(ConfigTree config) {
		this._config = config;

		ConfigTree[] configTree = _config.getChildren(MONITOR_VALUE);
		if (null == configTree || configTree.length < 1) {
		    logger.warn("Missing or empty destination list - This action class won't have any effect");
                    return;        
                }

		list = new ArrayList<MonitoringPatternBean>();
        
		for (ConfigTree curr : configTree) {
		    try
		    {
                        String pattern = curr.getAttribute(MVEL_PATTERN, "");
                        String name = curr.getAttribute(PATTERN_NAME, "");
                        String conditional = curr.getAttribute(PATTERN_CONDITIONAL, "");
                        list.add(new MonitoringPatternBean(pattern, name, conditional));
		    }
                    catch (Exception e)
                    {
                        logger.error("Could not form MonitoringPatternBean from configuration ", e);
                        throw new RuntimeException(e);
                    }
            }
        
            mvelMonitor = new MVELMonitor(config);        
            try {
        	mvelMonitor.registerMBean();
            } catch (Exception e) {
                logger.error("Could not register MBean", e);
                throw new RuntimeException(e);
            }
	}
	
	public Message process(Message message) throws ActionProcessingException {
	    if (list != null) {
	        for (int i = 0; i<list.size(); i++) {
	            MonitoringPatternBean mpb = list.get(i);
	            String object = null;
	            ObjectMapper mapper = new ObjectMapper();
	            try {
			object = (String) mapper.getObjectFromMessage(message, mpb.getPattern());
			Interpreter inter = new Interpreter();
			inter.eval(BEANSHELL_VARIABLE + " = " + object + mpb.getConditional());
			Boolean b = (Boolean)inter.get(BEANSHELL_VARIABLE);
			if (b.booleanValue()) {
			    mvelMonitor.addEvent(message, mpb);
			}
	            } catch (ObjectMappingException e1) {
			throw new ActionProcessingException(e1);
	            } catch (EvalError e) {
	                throw new ActionProcessingException(e);
	            }
	        }
	    }			
	    return message;
	}
}
