package org.jboss.soa.esb.addressing;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

import java.net.URI;
import java.net.URISyntaxException;

/**
 * Represents an interaction pattern for a specific message exchange. When sending a message
 * the sender application can specify where errors are to be returned, where responses are
 * to go, along with other interaction information.
 * 
 * A call represents an exchange pattern for this message. It is built up as the message
 * flows through the ESB and identifies where the message should go, along with any
 * routing information for faults, replies etc.
 *
 * 
 * To: the destination. MANDATORY.
 * From: the sender. OPTIONAL. If not defined, then the sender MAY be inferred from the transport.
 * ReplyTo: the destination for any response. OPTIONAL.
 * FaultTo: the destination for any error message. OPTIONAL.
 * RelatesTo: used to indicate that this message is related to another. OPTIONAL.
 * Action: used by the sender to indicate the semantics of the message. Must be unique. MANDATORY.
 * MessageID: used to uniquely identify this message. OPTIONAL.
 * 
 * @author marklittle
 *
 */

// TODO maybe this should simply go into the Header interface?

public class Call
{
	/**
	 * Create a new (empty) call.
	 */
	
	public Call ()
	{
	}
	
	/**
	 * Copy constructor.
	 * 
	 * @param copy
	 */
	
	public Call (Call copy) throws URISyntaxException
	{
	    _to = ((copy._to == null) ? null : copy._to.copy());
	    _from = ((copy._from == null) ? null : copy._from.copy());
	    _replyTo = ((copy._replyTo == null) ? null : copy._replyTo.copy());
	    _relatesTo = ((copy._relatesTo == null) ? null : new URI(copy._relatesTo.toString()));
	    _faultTo = ((copy._faultTo == null) ? null : copy._faultTo.copy());
	    _action = ((copy._action == null) ? null : new URI(copy._action.toString()));
	    _messageID = ((copy._messageID == null) ? null : new URI(copy._messageID.toString()));
	}
	
	/**
	 * Create a new call, whose To field is set to the supplied EPR.
	 * 
	 * @param epr the To field.
	 */
	
	public Call (EPR epr)
	{
		_to = epr;
	}
	
	/**
	 * Set the To field. Must not be null.
	 * 
	 * @param epr the To field value.
	 */
	
	public void setTo (EPR epr)
	{
		_to = epr;
	}
	
	/**
	 * @return the To field.
	 */
	
	public EPR getTo ()
	{
		return _to;
	}
	
	/**
	 * Set the From field. May be null.
	 * 
	 * @param from the value of the field.
	 */
	
	public void setFrom (EPR from)
	{
		_from = from;
	}
	
	/**
	 * @return the From field.
	 * @throws URISyntaxException thrown if the address is invalid.
	 */
	
	public EPR getFrom ()
	{
		return _from;
	}
	
	/**
	 * Set the ReplyTo field. May be null.
	 * 
	 * @param replyTo the value of the field.
	 */
	
	public void setReplyTo (EPR replyTo)
	{
		_replyTo = replyTo;
	}
	
	/**
	 * @return the ReplyTo field.
	 * @throws URISyntaxException thrown if the address is invalid.
	 */
	
	public EPR getReplyTo ()
	{
		return _replyTo;
	}
	
	/**
	 * Set the FaultTo field. May be null.
	 * 
	 * @param uri the value of the field.
	 */
	
	public void setFaultTo (EPR uri)
	{
		_faultTo = uri;
	}
	
	/**
	 * @return the FaultTo field.
	 */
	
	public EPR getFaultTo ()
	{
		return _faultTo;
	}
	
	/**
	 * Set the RelatesTo field.
	 * 
	 * @param uri the value to set.
	 */
	
	public void setRelatesTo (URI uri)
	{
		_relatesTo = uri;
	}
	
	/**
	 * @return the RelatesTo field.
	 */
	
	public URI getRelatesTo ()
	{
		return _relatesTo;
	}
	
	/**
	 * Set the Action field.
	 * @param uri the value to set.
	 */
	
	public void setAction (URI uri)
	{
		_action = uri;
	}

	/**
	 * @return the Action field.
	 */
	public URI getAction ()
	{
		return _action;
	}
	
	/**
	 * Set the MessageID for this instance.
	 * 
	 * @param uri the value to use.
	 */
	
	public void setMessageID (URI uri)
	{
		_messageID = uri;
	}
	
	/**
	 * @return the MessageID field.
	 */
	
	public URI getMessageID ()
	{
		return _messageID;
	}
	
	public final boolean empty ()
	{
		if ((_to == null) && (_from == null) && (_replyTo == null))
		{
			if ((_relatesTo == null) && (_faultTo == null) && (_action == null) && (_messageID == null))
				return true;
		}
		
		return false;
	}
	
	/**
	 * Copy the instance specified. This does a shallow (ptr) copy, so is not
	 * recommended.
	 * 
	 * @param from the instance to copy.
	 * @deprecated use the copy constructor.
	 */
	
	public void copy (Call from)
	{
		Call fromImpl = (Call) from;
		
		_to = fromImpl._to;
		_from = fromImpl._from;
		_replyTo = fromImpl._replyTo;
		_relatesTo = fromImpl._relatesTo;
		_faultTo = fromImpl._faultTo;
		_action = fromImpl._action;
		_messageID = fromImpl._messageID;
	}
	
//	public void setMetaData (MetaData md);
//	public MetaData getMetaData ();
//	public void addReferenceParameter (...);
	
	/**
	 * Prints the entire contents of the Call, even if most of the fields are null.
	 */
	
	public String stringForm ()
	{
		return "To: "+_to+", From: "+_from+", ReplyTo: "+_replyTo+", FaultTo: "+_faultTo+", Action: "+_action+", MessageID: "+_messageID+", RelatesTo: "+_relatesTo;
	}
	
	/**
	 * Print the content of this header, ignoring all null fields.
	 */
	
	public String toString ()
	{
		String result = "";
		
		if (_to != null)
			result = "To: "+_to;
		if (_from != null)
			result = result + " From: "+_from;
		if (_replyTo != null)
			result = result + " ReplyTo: "+_replyTo;
		if (_faultTo != null)
			result = result + " FaultTo: "+_faultTo;
		if (_action != null)
			result = result + " Action: "+_action;
		if (_messageID != null)
			result = result + " MessageID: "+_messageID;
		if (_relatesTo != null)
			result = result + " RelatesTo: "+_relatesTo;
		
		return result;
	}
	
	/**
	 * This instance is valid if all mandatory elements are set.
	 * 
	 * @return <code>true</code> if all mandatory elements are set, <code>false</code> otherwise.
	 */
	
	public boolean valid ()
	{
		if ((_to != null) && (_action != null))
			return true;
		else
			return false;
	}
	
	private EPR _to = null;
	private EPR _from = null;
	private EPR _faultTo = null;
	private EPR _replyTo = null;
	private URI _relatesTo = null;
	private URI _action = null;
	private URI _messageID = null;
	
} 
