package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * This class represents the endpoint reference for services.
 */

import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using FTPS style EPRs.
 * 
 * @author marklittle
 */
public class FTPSEpr extends FTPEpr
{
	public static final String CERTIFICATE_TAG = "certificate";
	public static final String CERTIFICATE_NAME_TAG = "certificateName";

	public FTPSEpr(EPR epr)
	{
		super(epr);
		
		if (epr instanceof FTPSEpr)
		{
		    certificateSet = ((FTPSEpr) epr).certificateSet;
		    certificateNameSet = ((FTPSEpr) epr).certificateNameSet;
		}
	}
	
	public FTPSEpr (EPR epr, Element header)
	{
		super(epr, header);
		
		NodeList nl = header.getChildNodes();

		for (int i = 0; i < nl.getLength(); i++)
		{
			String prefix = nl.item(i).getPrefix();
			String tag = nl.item(i).getLocalName();
			
			try
			{
				if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
				{
					if ((tag != null) && (tag.equals(CERTIFICATE_TAG)))
					{
						getAddr().addExtension(CERTIFICATE_TAG, nl.item(i).getTextContent());
						certificateSet = true;
					}
					else if ((tag != null) && (tag.equals(CERTIFICATE_NAME_TAG)))
					{
						setCertificateName(nl.item(i).getTextContent());
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}

	public FTPSEpr(URI uri) throws URISyntaxException
	{
		super(uri, type().toString());
	}
	
	public FTPSEpr(URI uri, URI cert, final String name) throws URISyntaxException
	{
		this(uri);
		
		if (cert != null)
		{
			setCertificateURI(cert);
		}
		if (name != null)
		{
			setCertificateName(name);
		}
	}

	public FTPSEpr(URI uri, String cert, final String name) throws URISyntaxException
	{
		this(uri);

		if (cert != null)
		{
			setCertificateURI(new URI(cert));
		}
		if (name != null)
		{
			setCertificateName(name);
		}
	}

	public FTPSEpr(String uri, String cert, final String name) throws URISyntaxException
	{
		this(new URI(uri));

		if(cert != null)
		{
			setCertificateURI(new URI(cert));
		}
		if (name != null)
		{
			setCertificateName(name);
		}
	}

	/**
	 * Set the Certificate URI for this endpoint.
	 * 
	 * @param cert
	 *            the address.
	 * @throws URISyntaxException
	 *             thrown if this EPR is malformed.
	 */

	public final void setCertificateURI(URI cert) throws URISyntaxException
	{
		if (cert == null)
			throw new IllegalArgumentException();

		if (certificateSet)
			throw new IllegalStateException("Certificate URI already set.");

		getAddr().addExtension(CERTIFICATE_TAG, cert.toASCIIString());
		certificateSet = true;
	}

	/**
	 * @return the certificate URI for this EPR.
	 * @throws URISyntaxException
	 *             thrown if this EPR is malformed.
	 */

	public final URI getCertificateURI () throws URISyntaxException
	{
		String cert = getAddr().getExtensionValue(CERTIFICATE_TAG);

		if (cert != null)
			return new URI(cert);
		else
			return null;
	}

	/**
	 * Set the Certificate Name for this endpoint.
	 * 
	 * @param name
	 *            the certificate name.
	 */

	public final void setCertificateName(final String name)
	{
		if (name == null)
			throw new IllegalArgumentException();

		if (certificateNameSet)
			throw new IllegalStateException("Certificate Name already set.");

		getAddr().addExtension(CERTIFICATE_NAME_TAG, name);
		certificateNameSet = true;
	}

	/**
	 * @return the certificate Name for this EPR.
	 * @throws URISyntaxException
	 *             thrown if this EPR is malformed.
	 */

	public final String getCertificateName()
	{
		return getAddr().getExtensionValue(CERTIFICATE_NAME_TAG);
	}

	public EPR copy ()
	{
	    return new FTPSEpr(this);
	}
	
	public String toString ()
	{
		return "FTPSEpr [ "+super.getAddr().extendedToString()+" ]";
	}
	
	public static URI type ()
	{
		return _type;
	}

	private boolean certificateSet = false;
	private boolean certificateNameSet ;

	private static URI _type;

	static
	{
		try
		{
			_type = new URI("urn:jboss/esb/epr/type/ftps");
		}
		catch (Exception ex)
		{
			ex.printStackTrace();

			throw new ExceptionInInitializerError(ex.toString());
		}
	}
}
