package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */


/**
 * This class represents the endpoint reference for services.
 */

import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using database style EPRs. Simply create instances of this
 * class instead of the base EPR.
 * 
 * @author marklittle
 *
 */

public class JDBCEpr extends EPR
{
	public static final String JDBC_PROTOCOL = "jdbc";
	public static final String DATASOURCE_TAG = "datasource";
	public static final String USERNAME_TAG = "username";
	public static final String PASSWORD_TAG = "password";
	public static final String SQL_TAG = "sql";
	public static final String DRIVER_TAG = "driver";
	public static final String URL_TAG = "URL";

	public static final String TABLE_NAME_TAG 			= "tablename";
	public static final String ORDER_BY_TAG				= "orderBy";
	public static final String MESSAGE_ID_COLUMN_TAG 	= "message_id_column";
	public static final String STATUS_COLUMN_TAG 		= "status_column";
    public static final String RETRY_COUNT_COLUMN_TAG 	= "retry_count_column";
	public static final String DATA_COLUMN_TAG 			= "message_column";
	public static final String TIMESTAMP_COLUMN_TAG 	= "insert_timestamp_column";
	public static final String WHERE_CONDITION_TAG 		= "whereCondition";
	public static final String POST_DEL_TAG				= "postDelete";
	public static final String ERROR_DEL_TAG			= "errorDelete";
	
	public static final String DEFAULT_REPLY_TO_TABLE_SUFFIX = "_reply_table";
	
	public JDBCEpr (EPR epr)
	{
		super(epr);
		
		if (epr instanceof JDBCEpr)
		{
		    JDBCEpr instance = (JDBCEpr) epr;
		
		    userSet = instance.userSet;
		    passwordSet = instance.passwordSet;
		    sqlSet = instance.sqlSet;
		    driverSet = instance.driverSet;
		    tableNameSet = instance.tableNameSet;
		    messageIdSet = instance.messageIdSet;
		    statusSet = instance.statusSet;
		    dataSet = instance.dataSet;
		    timestampSet = instance.timestampSet;
		}
	}
	
	public JDBCEpr (EPR epr, Element header)
	{
		super(epr);
		
		NodeList nl = header.getChildNodes();

		for (int i = 0; i < nl.getLength(); i++)
		{
			try
			{
				String prefix = nl.item(i).getPrefix();
				String tag = nl.item(i).getLocalName();
				
				if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
				{
					if (tag != null)
					{
						if (tag.equals(USERNAME_TAG))
						    setUserName(nl.item(i).getTextContent());
						else
						{
							if (tag.equals(PASSWORD_TAG))
							    setPassword(nl.item(i).getTextContent());
							else
							{
								if (tag.equals(SQL_TAG))
								    setSQL(nl.item(i).getTextContent());
								else
								{
									if (tag.equals(DRIVER_TAG))
									    setDriver(nl.item(i).getTextContent());
									else
									{
										if (tag.equals(URL_TAG))
										    setURL(nl.item(i).getTextContent());
										else
										{
											if (tag.equals(TABLE_NAME_TAG))
											    setTableName(nl.item(i).getTextContent());
											else
											{
												if (tag.equals(MESSAGE_ID_COLUMN_TAG))
												    setMessageIdColumn(nl.item(i).getTextContent());
												else
												{
													if (tag.equals(STATUS_COLUMN_TAG))
													    setStatusColumn(nl.item(i).getTextContent());
													else
													{
														if (tag.equals(DATA_COLUMN_TAG))
														    setDataColumn(nl.item(i).getTextContent());
														else
														{
															if (tag.equals(TIMESTAMP_COLUMN_TAG))
															    setTimestampColumn(nl.item(i).getTextContent());
															else
															{
																if (tag.equals(POST_DEL_TAG))
																	getAddr().addExtension(POST_DEL_TAG, nl.item(i).getTextContent());
																else
																{
																	if (tag.equals(ERROR_DEL_TAG))
																		getAddr().addExtension(ERROR_DEL_TAG, nl.item(i).getTextContent());
                                                                    else
                                                                    {
                                                                        if (tag.equals(RETRY_COUNT_COLUMN_TAG))
                                                                            setRetryCountColumn(nl.item(i).getTextContent());
                                                                    }
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			catch (Exception ex)
			{
				ex.printStackTrace();
			}
		}
	}			
	
	public JDBCEpr (String url, String sql) throws URISyntaxException
	{
		this(new URI(url));
		
		setSQL(sql);
	}

	public JDBCEpr (String url) throws URISyntaxException
	{
		this(url,true,true);
	}

	public JDBCEpr (String url, boolean postDelete, boolean errorDelete) throws URISyntaxException
	{
		this(new URI(url));
		if (postDelete)
			getAddr().addExtension(POST_DEL_TAG,Boolean.toString(postDelete));
		if (errorDelete)
			getAddr().addExtension(ERROR_DEL_TAG,Boolean.toString(errorDelete));
	}

	protected JDBCEpr(final URI uri)
	{
		super(uri, type().toString());
	}

	public EPR copy ()
	{
	    return new JDBCEpr(this);
	}

	/**
	 * Set the URL for this endpoint.
	 * 
	 * @param url the address.
	 */
	
	public final void setURL (String url)
	{
		super.setAddr(new PortReference(url));
		setType(type().toString()) ;
	}
	
	/**
	 * Get the URL address.
	 * 
	 * @return the address.
	 */
	
	public final String getURL ()
	{
		return getAddr().getAddress();
	}

	/**
	 * Set the username for this FTP EPR.
	 * 
	 * @param username the user's name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setUserName (String username) throws URISyntaxException
	{
		if (username == null)
			throw new IllegalArgumentException();
		
		if (userSet)
			throw new IllegalStateException("Username already set.");
		
		getAddr().addExtension(USERNAME_TAG, username);
		userSet = true;
	}
	
	/**
	 * @return the user's name associated with this EPR.
	 */
	
	public final String getUserName ()
	{
		return getAddr().getExtensionValue(USERNAME_TAG);
	}
	
	/**
	 * Set the password for this FTP EPR.
	 * 
	 * @param password the user's name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setPassword (String password) throws URISyntaxException
	{
		if (password == null)
			throw new IllegalArgumentException();
		
		if (passwordSet)
			throw new IllegalStateException("Password already set.");
		
		getAddr().addExtension(PASSWORD_TAG, password);
		passwordSet = true;
	}
	
	/**
	 * @return the password associated with this EPR.
	 */
	
	public final String getPassword ()
	{
		return getAddr().getExtensionValue(PASSWORD_TAG);
	}
	
	/**
	 * Set the JNDI datasource value that is used by this EPR.
	 * @param datasource JNDI datasource
	 */
	public final void setDatasource (String datasource) {
		getAddr().addExtension(DATASOURCE_TAG, datasource);
	}
	
	/**
	 * Get the JNDI datasource value that is used by this EPR.
	 * @return datasource
	 */
	public final String getDatasource() {
		return getAddr().getExtensionValue(DATASOURCE_TAG);
	}
	
	/**
	 * Set the SQL command that is used by this EPR.
	 * 
	 * @param sql the statement.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setSQL (String sql) throws URISyntaxException
	{
		if (sql == null)
			throw new IllegalArgumentException();
		
		if (sqlSet)
			throw new IllegalStateException("SQL already set.");
		
		getAddr().addExtension(SQL_TAG, sql);
		sqlSet = true;
	}
	
	/**
	 * @return the SQL statement for this EPR.
	 */
	
	public final String getSQL ()
	{
		return getAddr().getExtensionValue(SQL_TAG);
	}
	
	/**
	 * Set the driver that is used by this EPR.
	 * 
	 * @param driver the driver.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setDriver (String driver) throws URISyntaxException
	{
		if (driver == null)
			throw new IllegalArgumentException();
		
		if (driverSet)
			throw new IllegalStateException("Driver already set.");
		
		getAddr().addExtension(DRIVER_TAG, driver);
		driverSet = true;
	}
	
	/**
	 * @return the driver used by this EPR.
	 */
	
	public final String getDriver ()
	{
		return getAddr().getExtensionValue(DRIVER_TAG);
	}
	
	/**
	 * Set the tablename that is used by this EPR.
	 * 
	 * @param tableName the table name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setTableName (String tableName) throws URISyntaxException
	{
		if (tableName == null)
			throw new IllegalArgumentException();
		
		if (tableNameSet)
			throw new IllegalStateException("Table name already set.");
		
		getAddr().addExtension(TABLE_NAME_TAG, tableName);
		tableNameSet = true;
	}
	
	/**
	 * @return the table name used by this EPR.
	 */
	
	public final String getTableName ()
	{
		return getAddr().getExtensionValue(TABLE_NAME_TAG);
	}
		
	/**
	 * Set the message id column name that is used by this EPR.
	 * 
	 * @param columnName the column name for the message ID.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setMessageIdColumn (String columnName) throws URISyntaxException
	{
		if (columnName == null)
			throw new IllegalArgumentException();
		
		if (messageIdSet)
			throw new IllegalStateException("Message Id column already set.");
		
		getAddr().addExtension(MESSAGE_ID_COLUMN_TAG, columnName);
		messageIdSet = true;
	}
	
	/**
	 * @return the message id column used by this EPR.
	 */
	
	public final String getMessageIdColumn ()
	{
		return getAddr().getExtensionValue(MESSAGE_ID_COLUMN_TAG);
	}

    /**
     * Set the retry count column name that is used by this EPR.
     *
     * @param columnName the column name for the retry count.
     * @throws URISyntaxException thrown if this EPR is malformed.
     */

    public final void setRetryCountColumn (String columnName) throws URISyntaxException
    {
        if(columnName == null) {
            return;
        }

        if (retryCountSet)
            throw new IllegalStateException("Retry Count column already set.");

        getAddr().addExtension(RETRY_COUNT_COLUMN_TAG, columnName);
        retryCountSet = true;
    }

    /**
     * @return the retry count column used by this EPR.
     */

    public final String getRetryCountColumn ()
    {
        return getAddr().getExtensionValue(RETRY_COUNT_COLUMN_TAG);
    }

	/**
	 * Set the status column that is used by this EPR.
	 * 
	 * @param statusColumn the status column.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setStatusColumn (String statusColumn) throws URISyntaxException
	{
		if (statusColumn == null)
			throw new IllegalArgumentException();
		
		if (statusSet)
			throw new IllegalStateException("Status column already set.");
		
		getAddr().addExtension(STATUS_COLUMN_TAG, statusColumn);
		statusSet = true;
	}
	
	/**
	 * @return the status column name used by this EPR.
	 */
	
	public final String getStatusColumn ()
	{
		return getAddr().getExtensionValue(STATUS_COLUMN_TAG);
	}
	
	/**
	 * Set the column that is used by this EPR to store message data.
	 * 
	 * @param dataColumn the column name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setDataColumn (String dataColumn) throws URISyntaxException
	{
		if (dataColumn == null)
			throw new IllegalArgumentException();
		
		if (dataSet)
			throw new IllegalStateException("Data column already set.");
		
		getAddr().addExtension(DATA_COLUMN_TAG, dataColumn);
		dataSet = true;
	}
	
	/**
	 * @return the data column name used by this EPR.
	 */
	
	public final String getDataColumn ()
	{
		return getAddr().getExtensionValue(DATA_COLUMN_TAG);
	}
	
	/**
	 * Set the column that is used by this EPR to store timestamp when the message was inserted.
	 * 
	 * @param timeColumn the column name.
	 * @throws URISyntaxException thrown if this EPR is malformed.
	 */
	
	public final void setTimestampColumn (String timeColumn) throws URISyntaxException
	{
		if (timeColumn == null)
			throw new IllegalArgumentException();
		
		if (timestampSet)
			throw new IllegalStateException("Timestamp column already set.");
		
		getAddr().addExtension(TIMESTAMP_COLUMN_TAG, timeColumn);
		timestampSet = true;
	}
	
	/**
	 * @return the timestamp column name used by this EPR.
	 */
	
	public final String getTimestampColumn ()
	{
		return getAddr().getExtensionValue(TIMESTAMP_COLUMN_TAG);
	}
	
	
	public final String getPostDelete()
	{
	    return getAddr().getExtensionValue(POST_DEL_TAG);
	}
	
	public final String getErrorDelete()
	{
	    return getAddr().getExtensionValue(ERROR_DEL_TAG);
	}
	
	public String toString ()
	{
		return "JDBCEpr [ "+super.getAddr().extendedToString()+" ]";
	}
	
	public static final URI type ()
	{
	    return _type;
	}
	
	protected final void changeTableName (String tableName) throws URISyntaxException
	{
	    tableNameSet = false;
	    
	    super.getAddr().removeExtension(TABLE_NAME_TAG, getTableName());
	    
	    setTableName(tableName);
	}
	
	private boolean userSet = false;
	private boolean passwordSet = false;
	private boolean sqlSet = false;
	private boolean driverSet = false;
	private boolean tableNameSet = false;
	private boolean messageIdSet = false;
    private boolean retryCountSet = false;
	private boolean statusSet = false;
	private boolean dataSet = false;
	private boolean timestampSet = false;
	
	private static URI _type;
	
	static
	{
	    try
		{
		    _type = new URI("urn:jboss/esb/epr/type/jdbc");
		}
		catch (Exception ex)
		{
		    ex.printStackTrace();
		    
		    throw new ExceptionInInitializerError(ex.toString());
		}
	}
	
} 
