/*
	Milyn - Copyright (C) 2006 - 2010

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License (version 2.1) as published by the Free Software
	Foundation.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

	See the GNU Lesser General Public License for more details:
	http://www.gnu.org/licenses/lgpl.txt
*/
package org.jboss.soa.esb.configure;

import org.apache.commons.logging.*;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.annotation.Destroy;
import org.jboss.soa.esb.lifecycle.annotation.Initialize;
import org.milyn.util.ClassUtil;

import java.lang.annotation.*;
import java.lang.reflect.*;
import java.util.*;

/**
 * Utility class for processing configuration annotations on an
 * ESB Component class.
 * <p/>
 * Code donated from Smooks project.
 *
 * @author <a href="mailto:tom.fennelly@gmail.com">tom.fennelly@gmail.com</a>
 */
public class Configurator {

    private static Log logger = LogFactory.getLog(Configurator.class);
    private static final Map<Class<?>, Class<?>> primitiveToObjectMap;

    static {
        primitiveToObjectMap = new HashMap<Class<?>, Class<?>>();
        primitiveToObjectMap.put(Integer.TYPE, Integer.class);
        primitiveToObjectMap.put(Long.TYPE, Long.class);
        primitiveToObjectMap.put(Boolean.TYPE, Boolean.class);
        primitiveToObjectMap.put(Float.TYPE, Float.class);
        primitiveToObjectMap.put(Double.TYPE, Double.class);
        primitiveToObjectMap.put(Character.TYPE, Character.class);
        primitiveToObjectMap.put(Short.TYPE, Short.class);
    }

    /**
     * Configure the supplied object instance using the supplied
     * {@link ConfigTree} instance.
     * @param instance The instance to be configured.
     * @param config The configuration.
     * @return The configured object instance.
     * @throws ConfigurationException Invalid field annotations.
     */
    public static <U> U configure(U instance, ConfigTree config) throws ConfigurationException {
        AssertArgument.isNotNull(instance, "instance");
        AssertArgument.isNotNull(config, "config");

        // process the field annotations (@ConfigParam)...
        processFieldConfigParamAnnotations(instance.getClass(), instance, config);

        // process the method annotations (@ConfigParam)...
        processMethodConfigAnnotations(instance, config);

        return instance;
    }
    
    private static <U> void processFieldConfigParamAnnotations(Class<?> runtimeClass, U instance, ConfigTree config) throws ConfigurationException {
        Field[] fields = runtimeClass.getDeclaredFields();

        // Work back up the Inheritance tree first...
        Class<?> superClass = runtimeClass.getSuperclass();
        if(superClass != null) {
            processFieldConfigParamAnnotations(superClass, instance, config);
        }

        for (Field field : fields) {
            ConfigProperty configParamAnnotation = null;
            
            configParamAnnotation = field.getAnnotation(ConfigProperty.class);
            if(configParamAnnotation != null) {
                applyConfigParam(configParamAnnotation, field, field.getType(), instance, config);
            }
        }
    }

    private static <U> void checkPropertiesConfigured(Class<?> runtimeClass, U instance) throws ConfigurationException {
        Field[] fields = runtimeClass.getDeclaredFields();

        // Work back up the Inheritance tree first...
        Class<?> superClass = runtimeClass.getSuperclass();
        if(superClass != null) {
            checkPropertiesConfigured(superClass, instance);
        }

        for (Field field : fields) {
            String fieldName = field.getName();
            Object fieldValue;

            try {
                fieldValue = ClassUtil.getField(field, instance);
            } catch (IllegalAccessException e) {
                throw new ConfigurationException("Unable to get property field value for '" + getLongMemberName(field) + "'.", e);
            }

            if(fieldValue != null) {
                // It's set so no need to check anything....
                continue;
            }

            ConfigProperty configParamAnnotation = field.getAnnotation(ConfigProperty.class);
            if(configParamAnnotation == null) {
                // Check is there's a setter method for this property, with the @ConfigParam annotation
                // configured on it...
                String setterName = ClassUtil.toSetterName(fieldName);
                Method setterMethod = ClassUtil.getSetterMethod(setterName, runtimeClass, field.getType());

                if(setterMethod != null) {
                    configParamAnnotation = setterMethod.getAnnotation(ConfigProperty.class);
                }
            }

            if(configParamAnnotation != null) {
                // If it's required and not configured with a default value of AnnotationConstants.NULL_STRING, error... 
                if(configParamAnnotation.use() == ConfigProperty.Use.REQUIRED) {
                    // Property configured and it's required....

                    String defaultVal = configParamAnnotation.defaultVal();

                    // If there is no default (i.e. it's "UNASSIGNED"), we have an error...
                    if(defaultVal.equals(AnnotationUtil.UNASSIGNED)) {
                        throw new ConfigurationException("Property '" + fieldName + "' not configured on class " + instance.getClass().getName() + "'.");
                    }

                    // If the default is "NULL", just continue...
                    if(defaultVal.equals(AnnotationUtil.NULL_STRING)) {
                        continue;
                    }

                    // Decode the default and set it on the property...
                    Object value;
                    try {
                    	value = decodeValue(defaultVal, field.getType());
                    } catch (PropertyDecodeException e) {
                    	e.setConfiguredClass(instance.getClass());
                    	e.setPropertyName(fieldName);
                    	throw e;
                    }
                    try {
                        ClassUtil.setField(field, instance, value);
                    } catch (IllegalAccessException e) {
                        throw new ConfigurationException("Unable to set property field value for '" + getLongMemberName(field) + "'.", e);
                    }
                }
            }
        }
    }

	private static <U> void processMethodConfigAnnotations(U instance, ConfigTree config) throws ConfigurationException {
        Method[] methods = instance.getClass().getMethods();

        for (Method method : methods) {
            ConfigProperty configParamAnnotation = method.getAnnotation(ConfigProperty.class);
            if(configParamAnnotation != null) {
                Class params[] = method.getParameterTypes();

                if(params.length == 1) {
                    applyConfigParam(configParamAnnotation, method, params[0], instance, config);
                } else {
                    throw new ConfigurationException("Method '" + getLongMemberName(method) + "' defines a @ConfigParam, yet it specifies more than a single paramater.");
                }
            }
        }
    }

    private static <U> void applyConfigParam(ConfigProperty configParam, Member member, Class type, U instance, ConfigTree config) throws ConfigurationException {
        String name = configParam.name();
        String paramValue;

        // Work out the property name, if not specified via the annotation....
        if(AnnotationUtil.NULL_STRING.equals(name)) {
            // "name" not defined.  Use the field/method name...
            if(member instanceof Method) {
                name = getPropertyName((Method)member);
                if(name == null) {
                    throw new ConfigurationException("Unable to determine the property name associated with '" +
                            getLongMemberName(member)+ "'. " +
                            "Setter methods that specify the @ConfigParam annotation " +
                            "must either follow the Javabean naming convention ('setX' for propert 'x'), or specify the " +
                            "propery name via the 'name' parameter on the @ConfigParam annotation.");
                }
            } else {
                name = member.getName();
            }
        }
        paramValue = config.getAttribute(name);

        if(paramValue == null) {
            paramValue = configParam.defaultVal();
            if(AnnotationUtil.NULL_STRING.equals(paramValue)) {
                // A null default was assigned...
                String[] choices = configParam.choice();
                assertValidChoice(choices, name, AnnotationUtil.NULL_STRING);
                setMember(member, instance, null);
                return;
            } else if(AnnotationUtil.UNASSIGNED.equals(paramValue)) {
                // No default was assigned...
                paramValue = null;
            }
        }

        if(paramValue != null) {
            String[] choices = configParam.choice();

            assertValidChoice(choices, name, paramValue);

            Object value;
            try {
            	value = decodeValue(paramValue, type);
            } catch (PropertyDecodeException e) {
            	e.setConfiguredClass(instance.getClass());
            	e.setPropertyName(name);
            	throw e;
            }
            setMember(member, instance, value);
        } else if(configParam.use() == ConfigProperty.Use.REQUIRED) {
            throw new ConfigurationException("Property '" + name + "' not specified on configuration:\n" + config);
        }
    }

    private static void assertValidChoice(String[] choices, String name, String paramValue) throws ConfigurationException {
        if(choices == null || choices.length == 0) {
            throw new RuntimeException("Unexpected annotation default choice value.  Should not be null or empty.  Code may have changed incompatibly.");
        } else if(choices.length == 1 && AnnotationUtil.NULL_STRING.equals(choices[0])) {
            // A choice wasn't specified on the paramater config.
            return;
        } else {
            // A choice was specified. Check it against the value...
            for (String choice : choices) {
                if(paramValue.equals(choice)) {
                    return;
                }
            }
        }

        throw new ConfigurationException("Value '" + paramValue + "' for property '" + name + "' is invalid.  Valid choices for this property are: " + Arrays.asList(choices));
    }

    private static String getLongMemberName(Member field) {
        return field.getDeclaringClass().getName() + "#" + field.getName();
    }

    private static <U> void setMember(Member member, U instance, Object value) throws ConfigurationException {
        try {
            if(member instanceof Field) {
                ClassUtil.setField((Field)member, instance, value);
            } else {
                try {
                    setMethod((Method)member, instance, value);
                } catch (InvocationTargetException e) {
                    throw new ConfigurationException("Failed to set property configuration value on '" + getLongMemberName(member) + "'.", e.getTargetException());
                }
            }
        } catch (IllegalAccessException e) {
            throw new ConfigurationException("Failed to set property configuration value on '" + getLongMemberName(member) + "'.", e);
        }
    }

    private static <U> void setMethod(Method method, U instance, Object value) throws IllegalAccessException, InvocationTargetException {
        method.invoke(instance, value);
    }

    public static <U> void initialise(U instance, ConfigTree config) throws ActionLifecycleException {
        try {
			checkPropertiesConfigured(instance.getClass(), instance);
		} catch (ConfigurationException e) {
			throw new ActionLifecycleException("Cannot initialize object instance because the objects properties have not been configured.", e);
		}
        invoke(instance, Initialize.class, config);
    }

    public static <U> void destroy(U instance) throws ActionLifecycleException {
        invoke(instance, Destroy.class, null);
    }

    private static <U> void invoke(U instance, Class<? extends Annotation> annotation, ConfigTree config) throws ActionLifecycleException {
        Method[] methods = instance.getClass().getMethods();

        for (Method method : methods) {
            if(method.getAnnotation(annotation) != null) {
            	Object[] args = null;
            	
                if(method.getParameterTypes().length == 0) {
                	// Nothing
                } else if(method.getParameterTypes().length == 1 && method.getParameterTypes()[0] == ConfigTree.class && config != null) {
                	// Pass the configTree
                	args = new Object[] {config};
                } else {
                    logger.warn("Method '" + getLongMemberName(method) + "' defines an @" + annotation.getSimpleName() + " annotation on a paramaterized method.  This is not allowed!");
                }

                try {
                	method.invoke(instance, args);
                } catch (IllegalAccessException e) {
                	throw new ActionLifecycleException("Error invoking @" + annotation.getSimpleName() + " method '" + method.getName() + "' on class '" + instance.getClass().getName() + "'.", e);
                } catch (InvocationTargetException e) {
                	Throwable targetException = e.getTargetException();
                	if(targetException instanceof ActionLifecycleException) {
                		throw (ActionLifecycleException) targetException;
                	}
                	throw new ActionLifecycleException("Error invoking @" + annotation.getSimpleName() + " method '" + method.getName() + "' on class '" + instance.getClass().getName() + "'.", targetException);
                }
            }
        }
    }

    private static String getPropertyName(Method method) {
        if(!method.getName().startsWith("set")) {
            return null;
        }

        StringBuffer methodName = new StringBuffer(method.getName());

        if(methodName.length() < 4) {
            return null;
        }

        methodName.delete(0, 3);
        methodName.setCharAt(0, Character.toLowerCase(methodName.charAt(0)));

        return methodName.toString();
    }

	@SuppressWarnings("unchecked")
	private static Object decodeValue(String value, Class type) throws PropertyDecodeException {
		if(primitiveToObjectMap.containsKey(type)) {
			type = primitiveToObjectMap.get(type);
		} else if(type.isEnum()) {
	        try {
	            return Enum.valueOf(type, value.trim());
	        } catch(IllegalArgumentException e) {
	            throw new PropertyDecodeException("Failed to decode '" + value + "' as a valid Enum constant of type '" + type.getName() + "'.", e);
	        }			
		}
		
		try {
			Constructor<?> constructor = type.getConstructor(String.class);			
			return constructor.newInstance(value);
		} catch (SecurityException e) {
			throw new PropertyDecodeException("Security exception accessing single-arg String Constructor for property type '" + type.getName() + "'.", e);
		} catch (NoSuchMethodException e) {
			throw new PropertyDecodeException("No single-arg String Constructor for property type '" + type.getName() + "'.  You may need to perform decoding of this property from inside an @Initialize method.", e);
		} catch (IllegalArgumentException e) {
			throw new PropertyDecodeException("Exception invoking single-arg String Constructor for property type '" + type.getName() + "'.", e);
		} catch (InstantiationException e) {
			throw new PropertyDecodeException("Exception invoking single-arg String Constructor for property type '" + type.getName() + "'.", e);
		} catch (IllegalAccessException e) {
			throw new PropertyDecodeException("Exception invoking single-arg String Constructor for property type '" + type.getName() + "'.", e);
		} catch (InvocationTargetException e) {
			throw new PropertyDecodeException("Exception invoking single-arg String Constructor for property type '" + type.getName() + "'.", e.getTargetException());
		}
	}

}
