/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.http.configurators;

import java.util.Properties;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.params.HttpConnectionManagerParams;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.http.Configurator;
import org.jboss.soa.esb.http.ESBMultiThreadedHttpConnectionManager;

/**
 * HTTP Connection configurator.
 * <p/>
 * This {@link Configurator} is always applied to HttpClient instances created by the
 * {@link org.jboss.soa.esb.http.HttpClientFactory}.  Of course, if you wish to make the
 * connection configurations in a different way, you can always create a custom {@link Configurator}
 * that creates and attches a different {@link HttpConnectionManager} instance that has differnt
 * configurations.
 * <p/>
 * Properties:
 * <ul>
 *      <li><b>max-total-connections</b>: Maximum total number of connection for the HttpClient instance.</li>
 *      <li><b>max-connections-per-host</b>: Maximum connection per target host for the HttpClient instance.  Note that
 *           the default value for this configuration is 2.  So, configuring the "max-total-connections" property
 *           without also configuring this property will have little effect on performance for a single host
 *           because it will only ever open a max of 2 connections (as per the default). 
 *      </li>
 * </ul>
 * <p/>
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class Connection extends Configurator {

    /**
     * Max total connections.
     */
    public static final String MAX_TOTAL_CONNECTIONS = "max-total-connections";
    /**
     * Max connections per host.
     */
    public static final String MAX_CONNECTIONS_PER_HOST = "max-connections-per-host";

    public void configure(HttpClient httpClient, Properties properties) throws ConfigurationException {
        final HttpConnectionManager connectionManager = new ESBMultiThreadedHttpConnectionManager() ;
        final String maxTotalConnections = properties.getProperty(MAX_TOTAL_CONNECTIONS) ;
        final String maxConnectionsPerHost = properties.getProperty(MAX_CONNECTIONS_PER_HOST) ;

        connectionManager.setParams(new ESBHttpConnectionManagerParams(maxTotalConnections, maxConnectionsPerHost)) ;
        httpClient.setHttpConnectionManager(connectionManager);
    }
    
    /**
     * Per connection manager instance of params.
     * @author kevin
     */
    private static class ESBHttpConnectionManagerParams extends HttpConnectionManagerParams
    {
        /**
         * Serial version UID for this class.
         */
        private static final long serialVersionUID = 6586457118564628500L;
        /**
         * Maximum total connections.
         */
        private final Integer maxTotalConnections ;
        /**
         * Maximum connections per host.
         */
        private final Integer maxConnectionsPerHost ;
        
        /**
         * Construct the connection manager parameters.
         * @param totalConnections The maximum total connections or null if defaults.
         * @param connectionsPerHost The maximum connections per host or null if defaults.
         * @throws ConfigurationException For errors parsing parameters.
         */
        ESBHttpConnectionManagerParams(final String totalConnections, final String connectionsPerHost)
            throws ConfigurationException
        {
            if (totalConnections != null) {
                try {
                    maxTotalConnections = Integer.valueOf(totalConnections.trim()) ;
                } catch (final NumberFormatException nfe) {
                    throw new ConfigurationException("Invalid " + MAX_TOTAL_CONNECTIONS + " property: " + totalConnections) ;
                }
            } else {
                maxTotalConnections = null ;
            }
            
            if (connectionsPerHost != null) {
                try {
                    maxConnectionsPerHost = Integer.valueOf(connectionsPerHost.trim()) ;
                } catch (final NumberFormatException nfe) {
                    throw new ConfigurationException("Invalid " + MAX_CONNECTIONS_PER_HOST + " property: " + connectionsPerHost) ;
                }
            } else {
                maxConnectionsPerHost = null ;
            }
        }
        
        @Override
        /**
         * Get the maximum connections per host.
         * @param The current host configuration.
         * @return the maximum connections per host.
         */
        public int getMaxConnectionsPerHost(final HostConfiguration hostConfiguration)
        {
            if (maxConnectionsPerHost != null)
            {
                return maxConnectionsPerHost.intValue() ;
            }
            else
            {
                return super.getMaxConnectionsPerHost(hostConfiguration) ;
            }
        }
        
        @Override
        /**
         * Get the maximum total connections.
         * @return the maximum connections per host.
         */
        public int getMaxTotalConnections()
        {
            if (maxTotalConnections != null)
            {
                return maxTotalConnections.intValue() ;
            }
            else
            {
                return super.getMaxTotalConnections() ;
            }
        }
    }
}