/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.lifecycle;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.log4j.Logger;

/**
 * This class represents a lifecycle resource.
 */
public final class LifecycleResource<R>
{
    /**
     * The logger for this class.
     */
    private static Logger logger = Logger.getLogger(LifecycleResource.class);
    
    /**
     * The lifecycle resource factory.
     */
    private final LifecycleResourceFactory<R> factory ;
    
    /**
     * The lifecycle resource map.
     */
    private final Map<String, R> lifecycleResourceMap = new HashMap<String, R>() ;
    
    /**
     * The lifecycle resource lock.
     */
    private ReadWriteLock lifecycleResourceLock = new ReentrantReadWriteLock() ;
    
    /**
     * The lifecycle resource ClassLoader
     */
    private final ClassLoader resourceClassLoader ;

    /**
     * The destroyed flag.
     */
    private boolean destroyed ;
    
    /**
     * Construct a lifecycle resource using the specified factory.
     * @param factory The lifecycle resource factory.
     * @param priority The relative priority of the resource for cleanup.
     */
    public LifecycleResource(final LifecycleResourceFactory<R> factory, final int priority)
    {
        this.factory = factory ;
        resourceClassLoader = factory.getClass().getClassLoader() ;
        LifecycleResourceManager.getSingleton().registerResource(this, resourceClassLoader, priority) ;
    }
    
    /**
     * Construct a lifecycle resource using the specified factory.
     * @param factory The lifecycle resource factory.
     */
    public LifecycleResource(final LifecycleResourceFactory<R> factory)
    {
        this.factory = factory ;
        resourceClassLoader = factory.getClass().getClassLoader() ;
        LifecycleResourceManager.getSingleton().registerResource(this, resourceClassLoader) ;
    }
    
    /**
     * Get the lifecycle resource.
     * @return The lifecycle resource.
     */
    public R getLifecycleResource()
        throws LifecycleResourceException
    {
        final LifecycleResourceManager lifecycleResourceManager = LifecycleResourceManager.getSingleton() ;
        final String identity = lifecycleResourceManager.getIdentity() ;
        final Lock readLock = lifecycleResourceLock.readLock() ;
        readLock.lock() ;
        try
        {
            final R currentResource = lifecycleResourceMap.get(identity) ;
            if (currentResource != null)
            {
                return currentResource ;
            }
        }
        finally
        {
            readLock.unlock() ;
        }
        
        final Lock writeLock = lifecycleResourceLock.writeLock() ;
        writeLock.lock() ;
        try
        {
            if (destroyed && !lifecycleResourceManager.isActive(resourceClassLoader))
            {
                throw new LifecycleResourceException("Resource no longer Active") ;
            }
            final R resource = lifecycleResourceMap.get(identity) ;
            if (resource != null)
            {
                return resource ;
            }
            if (logger.isDebugEnabled())
            {
                logger.debug("Creating resource using factory: " + factory + " with identity " + identity) ;
            }
            final R newResource = factory.createLifecycleResource(identity) ;
            lifecycleResourceMap.put(identity, newResource) ;
            return newResource ;
        }
        finally
        {
            writeLock.unlock() ;
        }
    }
    
    /**
     * Cleanup the specified resource.
     * @param identity The identity of the resource.
     */
    void cleanupResource(final String identity)
    {
        final R resource ;
        
        final Lock writeLock = lifecycleResourceLock.writeLock() ;
        writeLock.lock() ;
        try
        {
            resource = lifecycleResourceMap.remove(identity) ;
        }
        finally
        {
            writeLock.unlock() ;
        }
        if (resource != null)
        {
            cleanupLifecycleResource(factory, resource, identity) ;
        }
    }
    
    /**
     * Destroy all remaining resources.
     */
    synchronized void destroyResources()
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Destroying all resources for factory : " + factory) ;
        }
        
        final Lock writeLock = lifecycleResourceLock.writeLock() ;
        writeLock.lock() ;
        try
        {
            final Iterator<Map.Entry<String, R>> entryIter = lifecycleResourceMap.entrySet().iterator() ;
            while(entryIter.hasNext())
            {
                final Map.Entry<String, R> entry = entryIter.next() ;
                final String identity = entry.getKey() ;
                final R resource = entry.getValue() ;
                cleanupLifecycleResource(factory, resource, identity) ;
            }
            lifecycleResourceMap.clear() ;
            destroyed = true ;
        }
        finally
        {
            writeLock.unlock() ;
        }
        
        if (logger.isDebugEnabled())
        {
            logger.debug("Destroyed all resources for factory : " + factory) ;
        }
    }
    
    /**
     * Destroy the lifecycle resource for the specific identity..
     * @param factory The lifecycle factory.
     * @param resource The lifecycle resource.
     * @param identity The identity.
     */
    private void cleanupLifecycleResource(final LifecycleResourceFactory<R> factory,
        final R resource, final String identity)
    {
        if (logger.isDebugEnabled())
        {
            logger.debug("Cleaning up resource using factory: " + factory + " with identity " + identity) ;
        }
        try
        {
            factory.destroyLifecycleResource(resource, identity) ;
        }
        catch (final Throwable th)
        {
            logger.warn("Unexpected error destroying lifecycle resource:" + th.getMessage()) ;
            if (logger.isDebugEnabled())
            {
                logger.debug("LifecycleResourceFactory: " + factory + ", identity: " + identity, th) ;
            }
        }
    }
}
