/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.config;


import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.management.JMException;
import javax.management.MBeanServer;

import org.jboss.deployment.DeploymentException;
import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.MainDeployerMBean;
import org.jboss.internal.soa.esb.listeners.war.Filter;
import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.listeners.war.WebDeploymentArchive;
import org.jboss.internal.soa.esb.listeners.war.WebModel;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.util.JBossDeployerUtil;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBResponseFilter;
import org.jboss.internal.soa.esb.webservice.ESBServiceContractReferencePublisher;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.mx.util.MBeanProxy;
import org.jboss.mx.util.MBeanProxyCreationException;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.config.WebGatewayBuilder;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;
import org.jboss.soa.esb.util.DeploymentArchive;
import org.jboss.system.ServiceMBeanSupport;

/**
 * Represents the JBoss AS 4 ESB deployment
 *
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @author <a href="mailto:mageshbk@jboss.com">Magesh Kumar B</a>
 * @version $Revision: 1.1 $
 */
public class JBoss4ESBDeployment extends ServiceMBeanSupport implements JBoss4ESBDeploymentMBean
{
   private URL document;
   private String jbossEsbXml;
   private String deploymentName;
   private String shortName;
   private List<ContractReferencePublisher> publishers;
   private List<Servlet> servlets;
   private ManagedLifecycleController controller;
   private ClassLoader classloader;
   private ClassLoader localCl;
   private File subDeployment;
   private ModelAdapter model;
   private static final String ESB_ARTIFACT_NAME = "jbossesb.esb";
   private static final String PREFIX_CANONICAL_NAME = "jboss.esb:deployment=";
   private File esbWarFiles;
   private File esbArchive;
   private URL url;

   public JBoss4ESBDeployment(URL document, final DeploymentInfo di, File esbWarFiles) throws DeploymentException
   {
      this.document = document;
      this.shortName = di.shortName;
      this.deploymentName = this.shortName;
      this.esbWarFiles = esbWarFiles;
      this.localCl = di.localCl;
      this.url = di.url;
      this.esbArchive = new File(di.watch.getFile());
      
      try
      {
         this.jbossEsbXml = JBossDeployerUtil.readEsbConfig(document.openStream());
         //This model is used only to add the action dependecies, later the model is recreated after every destroy/create cycle.
         this.model = JBossDeployerUtil.getJbossEsbModel(this.jbossEsbXml);
      }
      catch (Exception e)
      {
         throw new DeploymentException("Error during create of ESB Module: "
                 + url, e);
      }
   }

   public String getJbossEsbXml()
   {
      return jbossEsbXml;
   }

   public String getJbossEsbXmlAsHtml()
   {
      return jbossEsbXml.replace("<", "&lt;").replace(">", "&gt;");
   }

   public void setClassloader(ClassLoader classloader)
   {
      this.classloader = classloader;
   }

   public void setPublishers(final List<ContractReferencePublisher> publishers)
   {
       this.publishers = publishers ;
   }
   
   public void setServlets(final List<Servlet> servlets)
   {
	   this.servlets = servlets;
   }
   
   public List<Servlet> getServlets()
   {
	   return servlets;
   }

   public ManagedLifecycleController getController()
   {
      return controller;
   }
   
   public String getDeploymentName()
   {
       return deploymentName ;
   }
   
   public void setDeploymentName(final String deploymentName)
   {
       this.deploymentName = deploymentName ;
   }
   
   public List<ContractReferencePublisher> getPublishers()
   {
       return publishers ;
   }
   
   public ModelAdapter getModel()
   {
       return model;
   }

   @Override
   public void createService() throws DeploymentException, ConfigurationException, JMException
   {

      try
      {
         // Redo initialization as the properties could have changed
         this.jbossEsbXml = JBossDeployerUtil.readEsbConfig(document.openStream());
         this.model = JBossDeployerUtil.getJbossEsbModel(jbossEsbXml);
      }
      catch (Exception e)
      {
         throw new DeploymentException("Error during create of ESB Module: "
                 + url, e);
      }
      ClassLoader old = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         LifecycleResourceManager.getSingleton().associateDeployment(deploymentName) ;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(old);
      }

      WebGatewayBuilder builder = new WebGatewayBuilder(esbWarFiles, esbArchive, deploymentName, localCl, model);
      File warDeployment = builder.build();
      if (warDeployment != null)
      {
         try
         {
             DeploymentInfo subDi = new DeploymentInfo(warDeployment.toURL(), null, getServer());
             log.info("Deploying '" + warDeployment.getName() + "'");
             getMainDeployer().deploy(subDi);
             this.setPublishers(builder.getPublishers());
             this.setServlets(builder.getServlets());
             this.subDeployment = warDeployment;
         }
         catch (Throwable throwable)
         {
            if(warDeployment.exists())
            {
                log.info("Deleting '" + warDeployment.getName() + "'");
                warDeployment.delete();
            }
            throw new ConfigurationException("Error deploying '" + warDeployment.getAbsolutePath() + "'.", throwable);
         }
      }
   }

   @Override
   public void startService() throws Exception
   {
      ClassLoader old = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         controller = Configuration.create(jbossEsbXml, serviceName, publishers, servlets);
         controller.start();
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(old);
      }

   }

   @Override
   public void stopService() throws Exception
   {
      ClassLoader old = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         ServicePublisher.removeServicePublishers(controller);
         controller.stop();
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(old);
      }
   }

   @Override
   public void destroyService() throws Exception
   {
      if (subDeployment != null)
      {
         log.info("Undeploying '" + subDeployment.getName() + "'");
         getMainDeployer().undeploy(subDeployment.toURL());
         if(subDeployment.exists())
         {
            log.info("Deleting '" + subDeployment.getName() + "'");
            if(!subDeployment.delete())
            {
               log.debug("Failed to delete sub deployment '" + subDeployment.getAbsolutePath() + "'.");
            }
         }
      }

      ClassLoader old = Thread.currentThread().getContextClassLoader();
      try
      {
         Thread.currentThread().setContextClassLoader(classloader);
         LifecycleResourceManager.getSingleton().disassociateDeployment(deploymentName) ;
         LifecycleResourceManager.getSingleton().destroyResources() ;
      }
      finally
      {
         Thread.currentThread().setContextClassLoader(old);
      }
   }

   private MainDeployerMBean getMainDeployer() throws MBeanProxyCreationException
   {
      MBeanServer server = MBeanServerLocator.locateJBoss();
      MainDeployerMBean mainDeployer = (MainDeployerMBean)MBeanProxy.get(MainDeployerMBean.class, MainDeployerMBean.OBJECT_NAME, server);
      return mainDeployer;
   }
}
