/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers110;

import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.AbstractScheduledListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.Bus;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsProviderType;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.Provider;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FsListenerDocument.FsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.FtpListenerDocument.FtpListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.GroovyListenerDocument.GroovyListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JbrListenerDocument.JbrListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.HttpListenerDocument.HttpListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsBusDocument.JmsBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsJcaProviderDocument.JmsJcaProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JmsMessageFilterDocument.JmsMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.SqlBusDocument.SqlBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.SqlListenerDocument.SqlListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.SqlProviderDocument.SqlProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.ScheduledListenerDocument.ScheduledListener;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.message.MessageAwareListener;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * ESB Aware Listeners Configuration Generator.
 * <p/>
 * Generates a "ConfigTree" style configuration for the ESB Aware Listeners. 
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ESBAwareGenerator {

    private static final Logger logger = Logger.getLogger(ESBAwareGenerator.class);

    /**
	 * XMLBeans based configuration model instance.
	 */
	private XMLBeansModel model;
	
	/**
	 * Public constructor.
	 * @param model XMLBeans based configuration model instance.
	 */
	public ESBAwareGenerator(XMLBeansModel model) {
		this.model = model;
	}

	/**
	 * Generate the configuration DOM for the target Server.
	 * @return The configuration DOM.
	 * @throws ConfigurationException Error creating configuration. 
	 */
	public Document generate() throws ConfigurationException {
		Document doc = YADOMUtil.createDocument();
		Element root;
		
		doc.appendChild(doc.createComment("\n\tNOTE: DO NOT MODIFY\n\tThis file was auto-generated.\n"));
		root = YADOMUtil.addElement(doc, "jbossesb-listeners");
		root.setAttribute("parameterReloadSecs", model.getParameterReloadSecs());

        // Add an InVM listener for each service...
        List<Service> services = model.getServices();
        for (int i = 0; i < services.size(); i++) {
            Service service = services.get(i);
            if(model.exposesInVMListener(service)) {
                addInVMListener(root, service, i);
            }
        }
		
		// Iterate over all the ESB Aware listner configs and map them to the listener configuration...
		List<Listener> listeners = model.getESBAwareListeners();
		for(Listener listener : listeners) {
			addESBAwareConfig(root, listener);
		}
		
		return doc;
	}

    private void addInVMListener(Element root, Service service, int serviceIndex) throws ConfigurationException {
        Element listenerNode = YADOMUtil.addElement(root, "InVM-" + serviceIndex);
        Element properties = listenerNode.getOwnerDocument().createElement("properties");

        MapperUtil.mapListenerServiceAttributes(listenerNode, service, false);
        listenerNode.setAttribute(ListenerTagNames.LISTENER_CLASS_TAG, MessageAwareListener.class.getName());

        Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);
        eprNode.setAttribute(ListenerTagNames.PROTOCOL_TAG, InVMEpr.INVM_PROTOCOL);

        MapperUtil.mapProperties(service.getPropertyList(), properties);
        eprNode.setAttribute(ListenerTagNames.URL_TAG, InVMEpr.INVM_PROTOCOL + "://"
                + InVMEpr.createEncodedServiceId(service.getCategory(), service.getName())
                + "/" + YADOMUtil.getAttribute(properties, "inVMPassByValue", "false")
                + "?" + YADOMUtil.getAttribute(properties, "inVMLockStep", "false")
                + "#" + YADOMUtil.getAttribute(properties, "inVMLockStepTimeout", "10000"));

        listenerNode.setAttribute(ListenerTagNames.MAX_THREADS_TAG, YADOMUtil.getAttribute(properties, ListenerTagNames.MAX_THREADS_TAG, "1"));

        listenerNode.setAttribute(ListenerTagNames.TRANSACTED_TAG, Boolean.toString(getInVMTransacted(service))) ;
        YADOMUtil.removeEmptyAttributes(eprNode);

        ActionMapper.map(listenerNode, service, model);
    }
    
    /**
     * Try to find the InVM transacted value.  Check to see if the transacted attribute
     * has been explicitly set on the service.  If it has not been explicitly set then we
     * check to see if any of the listener/bus configurations are transacted and assume the
     * InVM is implicitly transacted.
     * 
     * @param service The current service.
     * @return true if transacted, false otherwise.
     */
    private boolean getInVMTransacted(final Service service)
        throws ConfigurationException
    {
        if (service.isSetInvmScope()) {
            return service.getInvmTransacted() ;
        } else if (service.isSetListeners()) {
            for (Listener listener: service.getListeners().getListenerList()) {
                if (listener instanceof ScheduledListener) {
                    if (((ScheduledListener)listener).getTransacted())
                        return true ;
                }
                else if (listener instanceof JmsListener) {
                    final Bus bus = model.getBus(listener.getBusidref()) ;
                    if (!(bus instanceof JmsBus))
                        continue ;
                    
                    final Provider provider = model.getProvider(bus) ;
                    if (provider instanceof JmsJcaProvider) {
                        // Jms JCA provider defaults to true
                        final JmsJcaProvider jmsJcaProvider = (JmsJcaProvider)provider ;
                        if (!jmsJcaProvider.isSetTransacted() || jmsJcaProvider.getTransacted())
                            return true ;
                    } else if (provider instanceof JmsProviderType) {
                        JmsMessageFilter filter = ((JmsListener)listener).getJmsMessageFilter() ;
                        if (filter == null)
                            filter = ((JmsBus)bus).getJmsMessageFilter() ;
                        if ((filter != null) && filter.getTransacted())
                            return true ;
                    }
                } else if (listener instanceof SqlListener) {
                    final Bus bus = model.getBus(listener.getBusidref()) ;
                    if (!(bus instanceof SqlBus))
                        continue ;
                    
                    final Provider provider = model.getProvider(bus) ;
                    if ((provider instanceof SqlProvider) && ((SqlProvider)provider).getTransacted())
                        return true ;
                }
            }
        }
        return false ;
    }

    /**
	 * Add a single ESB Aware Listener configuration node to configuration root. 
	 * @param root Configuration root node.
	 * @param listener The ESB Aware Listener configuration to be added.
	 * @throws ConfigurationException Invalid listener configuration.
	 */
	private void addESBAwareConfig(Element root, Listener listener) throws ConfigurationException {
		Element listenerConfigTree;
		Service listenerService;
		
		// Of course we could do the following reflectively if we really want to - TODO perhaps!!
		if(listener instanceof JmsListener) {
			listenerConfigTree = JmsListenerMapper.map(root, (JmsListener)listener, model);
        } else if (listener instanceof ScheduledListener) {
            listenerConfigTree = ScheduledListenerMapper.map(root, (ScheduledListener) listener, model);
        } else if(listener instanceof FsListener) {
         listenerConfigTree = FsListenerMapper.map(root, (FsListener)listener, model);
		} else if (listener instanceof FtpListener) {
			listenerConfigTree = FtpListenerMapper.map(root, (FtpListener)listener, model);
		} else if (listener instanceof SqlListener) {
			listenerConfigTree = SqlListenerMapper.map(root, (SqlListener)listener, model);
        } else if (listener instanceof JbrListener) {
            logger.error("JbossRemoting listener only supported in Gateway mode.");
            return;
        } else if (listener instanceof HttpListener) {
            logger.error("Http listener only supported in Gateway mode.");
            return;            
        } else if (listener instanceof GroovyListener) {
            listenerConfigTree = GroovyListenerMapper.map(root, (GroovyListener)listener, model);
		} else {
			listenerConfigTree = UntypedListenerMapper.map(root, listener, model);
		}

        if (listener instanceof AbstractScheduledListener) {
            ScheduleMapper.map(listenerConfigTree, (AbstractScheduledListener) listener, model);
        }

        // Locate the Service to which the listener is bound...
		listenerService = model.getService(listener);
		
		// Map the actions and notiications...
		ActionMapper.map(listenerConfigTree, listenerService, model);
		NotificationMapper.map(listenerConfigTree, listenerService, model);
	}
}
