/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers130;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.JmsProviderType;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.ActivationConfigDocument.ActivationConfig;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.JmsBusDocument.JmsBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.JmsJcaProviderDocument.JmsJcaProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.JmsMessageFilterDocument.JmsMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.JmsMessageFilterDocument.JmsMessageFilter.DestType;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.PropertyDocument.Property;
import org.jboss.soa.esb.listeners.gateway.JmsGatewayListener;
import org.jboss.soa.esb.listeners.gateway.PackageJmsMessageContents;
import org.jboss.soa.esb.listeners.jca.ActivationMapper;
import org.jboss.soa.esb.listeners.jca.JcaConstants;
import org.jboss.soa.esb.listeners.jca.JcaGatewayListener;
import org.jboss.soa.esb.listeners.jca.JcaJMSInflowMessageProcessorAdapter;
import org.jboss.soa.esb.listeners.jca.JcaJMSMessageAwareComposer;
import org.jboss.soa.esb.listeners.jca.JcaMessageAwareListener;
import org.jboss.soa.esb.util.ClassUtil;
import org.w3c.dom.Element;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Performs the mapping of a &lt;jms-listener&gt; XSD based configuration to the "ConfigTree"
 * style configuration, adding the "ConfigTree" listener config to the "root" node.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JmsListenerMapper {
	// The default JMS JCA adapter.
	private static final String DEFAULT_JMS_ADAPTER = "jms-ra.rar" ;

	private static Logger log = Logger.getLogger( JmsListenerMapper.class );

	/**
	 * Perform the mapping.
	 * @param root The "ConfigTree" configuration root node.
	 * @param listener The Jmslistener to be mapped into the ConfigTree.
	 * @param model The configuration model from which the mapping is being performed.
	 * @return The ConfigTree listener configuration node.
	 * @throws org.jboss.soa.esb.ConfigurationException Invalid listener configuration.
	 */
	public static Element map(Element root, JmsListener listener, XMLBeansModel model) throws ConfigurationException {
		Element listenerNode = YADOMUtil.addElement(root, "listener");
		JmsBus bus;
		JmsProviderType provider;

        listenerNode.setAttribute("name", listener.getName());

		try {
			bus = (JmsBus) model.getBus(listener.getBusidref());
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid busid reference [" + listener.getBusidref() + "] on listener [" + listener.getName() + "].  A <jms-listener> must reference a <jms-bus>.");
		}
		try {
			provider = (JmsProviderType) model.getProvider(bus);
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid bus config [" + listener.getBusidref() + "].  Should be contained within a <jms-provider> instance.  Unexpected exception - this should have caused a validation error!");
		}

		JmsMessageFilter messageFilter = listener.getJmsMessageFilter();
		if(messageFilter == null) {
			messageFilter = bus.getJmsMessageFilter();
			if(messageFilter == null) {
				throw new ConfigurationException("No <jms-detination> defined on either <jms-listener> [" + listener.getName() + "] or <jms-bus> [" + bus.getBusid() + "].");
			}
		}

		final Class<?> gatewayClass ;
		if (provider instanceof JmsJcaProvider) {
		    final JmsJcaProvider jmsJcaProvider = (JmsJcaProvider)provider ;
                    mapJmsJcaAttributes(listener, listenerNode, bus, jmsJcaProvider, messageFilter) ;
		    gatewayClass = JcaGatewayListener.class ;
		} else {
		    gatewayClass = JmsGatewayListener.class ;
		}

		// Map the standard listener attributes - common across all listener types...
		MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
		// Map the <property> elements targeted at the listener - from the listener itself.
		MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);
		if(listener.getIsGateway()) {

			listenerNode.setAttribute("gatewayClass", gatewayClass.getName());
			listenerNode.setAttribute(ListenerTagNames.IS_GATEWAY_TAG, Boolean.toString(listener.getIsGateway()));

			// Map EPR related attributes onto the listener - from the bus and provider and listener.
			// Note: This will change - the Gateways will also support the EPR element...
			mapJmsEprProperties(listenerNode, provider, bus, listener, messageFilter);
                        listenerNode.setAttribute(ListenerTagNames.PROTOCOL_TAG, JMSEpr.JMS_PROTOCOL);

			MapperUtil.mapEPRProperties(listener, listenerNode, model);
		} else {
			Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);


			// Map EPR related attributes onto the EPR - from the bus and provider and listener...
			mapJmsEprProperties(eprNode, provider, bus, listener, messageFilter);
			eprNode.setAttribute(ListenerTagNames.PROTOCOL_TAG, JMSEpr.JMS_PROTOCOL);

			MapperUtil.mapEPRProperties(listener, eprNode, model);
			// Remove any empty attributes set on the EPR config...
			YADOMUtil.removeEmptyAttributes(eprNode);
		}

		// Remove any empty attributes set on the listener config...
		YADOMUtil.removeEmptyAttributes(listenerNode);

		return listenerNode;
	}

    private static void mapJmsEprProperties(Element toElement, JmsProviderType provider, JmsBus bus, JmsListener listener, JmsMessageFilter messageFilter) throws ConfigurationException {
		if(messageFilter.getDestType() == DestType.QUEUE) {
            toElement.setAttribute(JMSEpr.DESTINATION_TYPE_TAG, JMSEpr.QUEUE_TYPE);
        } else {
            toElement.setAttribute(JMSEpr.DESTINATION_TYPE_TAG, JMSEpr.TOPIC_TYPE);
        }
        toElement.setAttribute(JMSEpr.DESTINATION_NAME_TAG, messageFilter.getDestName());
		toElement.setAttribute(JMSEpr.MESSAGE_SELECTOR_TAG, messageFilter.getSelector());
		toElement.setAttribute(JMSEpr.CONNECTION_FACTORY_TAG, provider.getConnectionFactory());
		toElement.setAttribute(JMSEpr.JNDI_CONTEXT_FACTORY_TAG, provider.getJndiContextFactory());
		toElement.setAttribute(JMSEpr.JNDI_PKG_PREFIX_TAG, provider.getJndiPkgPrefix());
		toElement.setAttribute(JMSEpr.JNDI_URL_TAG, provider.getJndiURL());
		toElement.setAttribute(JMSEpr.PERSISTENT_TAG, Boolean.toString( messageFilter.getPersistent()));
		toElement.setAttribute(JMSEpr.ACKNOWLEDGE_MODE_TAG, messageFilter.getAcknowledgeMode());
		toElement.setAttribute(JMSEpr.JMS_SECURITY_PRINCIPAL_TAG, messageFilter.getJmsSecurityPrincipal());
		toElement.setAttribute(JMSEpr.JMS_SECURITY_CREDENTIAL_TAG, messageFilter.getJmsSecurityCredential());
		toElement.setAttribute(JMSEpr.TRANSACTED_TAG, Boolean.toString( messageFilter.getTransacted()));
		
		String durableSubsName = listener.getDurableSubscriptionName();
		if(durableSubsName != null) {
			if(messageFilter.getDestType() != DestType.TOPIC) {
				throw new ConfigurationException("JMS listener configuration on JMS Bus '" + bus.getBusid() + "' defines a durable subscription name.  Durable subscribers are only supported for JMS Topics.");				
			}
			
			toElement.setAttribute(JMSEpr.DURABLE_SUBSCRIPTION_NAME, durableSubsName);
			
			if(listener.getClientId() != null) {
				toElement.setAttribute(JMSEpr.CLIENT_ID, listener.getClientId());
			} else {
				// If not defined on the listener, default the clientId to the configured listener name...
				toElement.setAttribute(JMSEpr.CLIENT_ID, listener.getName());
			}
		}		
	}

    private static void mapJmsJcaAttributes(final JmsListener listener,
        final Element listenerNode, final JmsBus bus, final JmsJcaProvider jmsJcaProvider,
        final JmsMessageFilter messageFilter)
        throws ConfigurationException {
        setAttribute(listenerNode, JcaConstants.ATTRIBUTE_ADAPTER,
            jmsJcaProvider.getAdapter(), DEFAULT_JMS_ADAPTER) ;
        setAttribute(listenerNode, JcaConstants.ATTRIBUTE_ENDPOINT_CLASS,
            jmsJcaProvider.getEndpointClass(), JcaJMSInflowMessageProcessorAdapter.class.getName()) ;
        setAttribute(listenerNode, JcaConstants.ATTRIBUTE_MESSAGING_TYPE,
            jmsJcaProvider.getMessagingType()) ;
        setAttribute(listenerNode, JcaConstants.ATTRIBUTE_JCA_BRIDGE,
            jmsJcaProvider.getJcaBridge()) ;
        final boolean transacted ;
        if (jmsJcaProvider.isSetTransacted())
        {
            transacted = jmsJcaProvider.getTransacted() ;
        }
        else
        {
            transacted = true ;
        }

        setAttribute(listenerNode, JcaConstants.ATTRIBUTE_TRANSACTED,
            Boolean.toString(transacted)) ;

        if (listener.getIsGateway())
        {
            setAttribute(listenerNode, ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG,
                PackageJmsMessageContents.class.getName()) ;
            setAttribute(listenerNode, ListenerTagNames.GATEWAY_COMPOSER_METHOD_TAG,
                "process") ;
        }
        else
        {
            setAttribute(listenerNode, ListenerTagNames.LISTENER_CLASS_TAG,
                JcaMessageAwareListener.class.getName()) ;
            setAttribute(listenerNode, JcaConstants.ATTRIBUTE_LISTENER_COMPOSER_CLASS,
                JcaJMSMessageAwareComposer.class.getName()) ;
        }

        final ActivationMapper activationMapper = getActivationMapper(listener, bus, jmsJcaProvider) ;
        final Map<String, String> activationConfigValues = new HashMap<String, String>() ;
        activationMapper.setDestination(activationConfigValues, messageFilter.getDestName()) ;
        activationMapper.setProviderAdapterJNDI(activationConfigValues, jmsJcaProvider.getProviderAdapterJNDI()) ;
        final int destType = messageFilter.getDestType().intValue() ;
        if (destType == DestType.INT_QUEUE)
        {
            activationMapper.setDestinationType(activationConfigValues, true) ;
        }
        else if (destType == DestType.INT_TOPIC)
        {
            activationMapper.setDestinationType(activationConfigValues, false) ;
        }
        else
        {
            throw new ConfigurationException("Unknown destination type: " + messageFilter.getDestType()) ;
        }

        activationMapper.setMessageSelector(activationConfigValues, messageFilter.getSelector()) ;
        final Integer maxThreads = (listener.isSetMaxThreads() ? Integer.valueOf(listener.getMaxThreads()) : null) ;
        activationMapper.setMaxThreads(activationConfigValues, maxThreads) ;
        
        final Element activationConfigElement = YADOMUtil.addElement(listenerNode, JcaConstants.ELEMENT_ACTIVATION_CONFIG) ;
        for(Map.Entry<String, String> entry: activationConfigValues.entrySet())
        {
            addPropertyElement(activationConfigElement, entry.getKey(), entry.getValue()) ;
        }

		final Set<String> defaultPropertyNames = activationConfigValues.keySet() ;
		log.debug("Default activation-config properties :" + defaultPropertyNames );

		/*
		 * 	add user activation-config properties specified in the provider element.
		 */
        ActivationConfig activationConfig = jmsJcaProvider.getActivationConfig();
        if ( activationConfig != null )
        {
            List<Property> propertyList = activationConfig.getPropertyList();
            for (Iterator<Property> iterator = propertyList.iterator(); iterator.hasNext();)
    		{
    			Property prop = iterator.next();
    			if ( defaultPropertyNames.contains( prop.getName() ))
        			throw new ConfigurationException( "activation-config already contains [" + prop.getName() + "], which cannot be overridden");

                addPropertyElement(activationConfigElement, prop.getName(), prop.getValue()) ;
    		}
        }
    }

    private static void setAttribute(final Element listenerNode,
        final String name, final String value)
    {
        if (value != null)
        {
            listenerNode.setAttribute(name, value) ;
        }
    }

    private static void setAttribute(final Element listenerNode,
        final String name, final String value, final String defaultValue)
    {
        if (value == null)
        {
            listenerNode.setAttribute(name, defaultValue) ;
        }
        else
        {
            listenerNode.setAttribute(name, value) ;
        }
    }

    private static void addPropertyElement(final Element activationConfigElement,
        final String name, final String value)
    {
        final Element propertyElement = YADOMUtil.addElement(activationConfigElement,
            JcaConstants.ELEMENT_PROPERTY) ;
        propertyElement.setAttribute("name", name) ;
        propertyElement.setAttribute("value", value) ;
    }

    private static ActivationMapper getActivationMapper(final JmsListener listener,
        final JmsBus bus, final JmsJcaProvider jmsJcaProvider)
        throws ConfigurationException
    {
        final String activationMapper = getActivationMapperClass(listener, bus, jmsJcaProvider); ;
        if (activationMapper == null)
        {
            throw new ConfigurationException("Could not locate ActivationMapper for adapter " + jmsJcaProvider.getAdapter()) ;
        }
        final Class activationMapperClass ;
        try
        {
            activationMapperClass = ClassUtil.forName(activationMapper.trim(), JmsListenerMapper.class) ;
        }
        catch (final ClassNotFoundException cnfe)
        {
            throw new ConfigurationException("Could not locate activation mapper class " + activationMapper, cnfe) ;
        }
        
        if (!ActivationMapper.class.isAssignableFrom(activationMapperClass))
        {
            throw new ConfigurationException("Activation mapper class " + activationMapper + " does not implement ActivationMapper interface") ;
        }
        try
        {
            return (ActivationMapper)(activationMapperClass.newInstance()) ;
        }
        catch (final Throwable th)
        {
            throw new ConfigurationException("Failed to instantiate activation mapper class " + activationMapper, th) ;
        }
    }
    
    private static String getActivationMapperClass(final JmsListener listener,
        final JmsBus bus, final JmsJcaProvider jmsJcaProvider)
        throws ConfigurationException
    {
        final String listenerActivationMapper = findProperty(listener.getPropertyList(), ListenerTagNames.JCA_ACTIVATION_MAPPER) ;
        if (listenerActivationMapper != null)
        {
            return listenerActivationMapper ;
        }
        
        final String busActivationMapper = findProperty(bus.getPropertyList(), ListenerTagNames.JCA_ACTIVATION_MAPPER) ;
        if (busActivationMapper != null)
        {
            return busActivationMapper ;
        }
        
        final String providerActivationMapper = findProperty(jmsJcaProvider.getPropertyList(), ListenerTagNames.JCA_ACTIVATION_MAPPER) ;
        if (providerActivationMapper != null)
        {
            return providerActivationMapper ;
        }
        
        final String adapter = (jmsJcaProvider.getAdapter() == null ? DEFAULT_JMS_ADAPTER : jmsJcaProvider.getAdapter()) ;
        
        final PropertyManager propertyManager = ModulePropertyManager.getPropertyManager(ModulePropertyManager.JCA_MODULE);
        final String activationMapper = propertyManager.getProperty("org.jboss.soa.esb.jca.activation.mapper." + adapter) ;
        if (activationMapper == null)
        {
            throw new ConfigurationException("Could not locate activation mapper for adapter " + adapter) ;
        }
        return activationMapper ;
    }
    
    private static String findProperty(final List<Property> properties, final String name)
    {
        if ((properties != null) && (name != null))
        {
            for(Property property: properties)
            {
                if (name.equals(property.getName()))
                {
                    return property.getValue() ;
                }
            }
        }
        return null ;
    }
}
