/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.management.JMException;
import javax.management.ObjectName;
import javax.management.MBeanServer;

import org.apache.log4j.Logger;
import org.jboss.beans.metadata.spi.BeanMetaData;
import org.jboss.beans.metadata.spi.builder.BeanMetaDataBuilder;
import org.jboss.deployers.client.plugins.deployment.AbstractDeployment;
import org.jboss.deployers.client.spi.DeployerClient;
import org.jboss.deployers.client.spi.DeploymentFactory;
import org.jboss.deployers.spi.attachments.MutableAttachments;
import org.jboss.deployers.spi.structure.ContextInfo;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.structure.spi.ClassLoaderFactory;
import org.jboss.deployers.vfs.spi.client.VFSDeployment;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.deployment.DeploymentException;
import org.jboss.internal.soa.esb.listeners.war.Filter;
import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.listeners.war.WebDeploymentArchive;
import org.jboss.internal.soa.esb.listeners.war.WebModel;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.util.JBossDeployerUtil;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBResponseFilter;
import org.jboss.internal.soa.esb.webservice.ESBServiceContractReferencePublisher;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.config.Configuration;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.config.ModelUtil;
import org.jboss.soa.esb.listeners.config.ServicePublisher;
import org.jboss.soa.esb.listeners.config.WebGatewayBuilder;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;
import org.jboss.soa.esb.util.DeploymentArchive;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * EsbDeployment for the Microcontainer. 
 * <p/> 
 * 
 * Microcontainer will call the lifecycle methods:
 * <lu>
 *  <li><i>create</i> will create the ESB deployment.</i>
 *  <li><i>start</i> will start the ESB deployment.</i>
 *  <li><i>stop</i> will stop the ESB deployment.</i>
 *  <li><i>destroy</i> will destroy the ESB deployment.</i>
 * </lu>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @author <a href="mailto:mageshbk@jboss.com">Magesh Kumar B</a>
 */
public class EsbDeployment implements EsbDeploymentMBean
{
    private Logger log = Logger.getLogger(EsbDeployment.class);
    
    private int state = CREATED;

    private String jbossEsbXml;
    private String deploymentName;
    private List<ContractReferencePublisher> publishers;
    private List<Servlet> servlets;
    private ManagedLifecycleController controller;
    private final String mbeanName;
    //private File subDeployment;
    private ClassLoader localCl;

    /** The meta data */
    private EsbMetaData esbMetaData;

    private VFSDeployment deployment;
    private DeploymentFactory factory = new DeploymentFactory();

    private DeployerClient mainDeployer;
    private File esbWarFiles;
    private File esbArchive;
        
    public EsbDeployment(final EsbMetaData esbMetaData, final String mbeanName, final VFSDeploymentUnit deploymentUnit, final File esbWarFiles) throws Exception
    {
        this.deploymentName = esbMetaData.getArchiveName();
        this.mbeanName = mbeanName;
        this.esbMetaData = esbMetaData;
        this.esbWarFiles = esbWarFiles;
        this.jbossEsbXml = JBossDeployerUtil.readEsbConfig(esbMetaData.getFile().openStream());
        this.esbArchive = new File(deploymentUnit.getRoot().toURL().getPath());
        this.localCl = deploymentUnit.getClassLoader();
    }
    
    public String getJbossEsbXml()
    {
        return jbossEsbXml;
    }

    public String getJbossEsbXmlAsHtml()
    {
        return jbossEsbXml.replace("<", "&lt;").replace(">", "&gt;");
    }

    public void setPublishers(final List<ContractReferencePublisher> publishers)
    {
        this.publishers = publishers;
    }
    
    public void setServlets(final List<Servlet> servlets)
    {
 	   this.servlets = servlets;
    }
    
    public List<Servlet> getServlets()
    {
 	   return servlets;
    }

    public void setMainDeployer(final DeployerClient mainDeployer)
    {
        this.mainDeployer = mainDeployer;
    }

    public ManagedLifecycleController getController()
    {
        return controller;
    }

    public String getDeploymentName()
    {
        return deploymentName;
    }

    public void setDeploymentName(final String deploymentName)
    {
        this.deploymentName = deploymentName;
    }

    public List<ContractReferencePublisher> getPublishers()
    {
        return publishers;
    }

    public void create() throws Exception
    {
    	state = CREATING;
    	
        // Redo initialization as the properties could have changed
        this.jbossEsbXml = JBossDeployerUtil.readEsbConfig(esbMetaData.getFile().openStream());
        esbMetaData.setModel(JBossDeployerUtil.getJbossEsbModel(jbossEsbXml));

        LifecycleResourceManager.getSingleton().associateDeployment(deploymentName);
        LifecycleResourceManager.deactivateHook();

        WebGatewayBuilder builder = new WebGatewayBuilder(esbWarFiles, esbArchive, deploymentName, localCl, esbMetaData.getModel());
        File subDeployment = builder.build();
        if (subDeployment != null)
        {
            VFSDeployment vfsDeployment = null;
            try
            {
                vfsDeployment = createVFSDeployment(subDeployment);
                mainDeployer.deploy(vfsDeployment);
                log.info("Deploying '" + vfsDeployment.getName() + "'");
                this.setPublishers(builder.getPublishers());
                this.setServlets(builder.getServlets());
                this.deployment = vfsDeployment;
            }
            catch (Throwable throwable)
            {
                if (vfsDeployment != null)
                {
                    VirtualFile deploymentFile = vfsDeployment.getRoot();
                    if(deploymentFile.exists())
                    {
                        log.info("Deleting '" + vfsDeployment.getSimpleName() + "'");
                        deploymentFile.delete();
                    }
                }
                throw new ConfigurationException("Error deploying '" + vfsDeployment.getName() + "'.", throwable);
            }
        }
        this.state = CREATED;
    }

    public void start() throws Exception
    {    
    	if ((state == STARTING) || (state == STARTED))
    		return;
    	state = STARTING;

    	try {
	        if (controller == null)
	        {
	        	log.info("Starting ESB Deployment '" + deploymentName + "'");
	            controller = Configuration.create(jbossEsbXml, new ObjectName(mbeanName), publishers, servlets);
	            controller.start();
	        }
	        state = STARTED;
    	} catch (Exception e) {
    		state = FAILED;
    		throw e;
    	}
    }

	public boolean isStarted() {
		return (controller != null);
	}

	public String getStateString() {
		return states[state];
	}
	
	public int getState() {
		return state;
	}
	
    public void stop() throws Exception
    {        
    	if (state != STARTED)
    		return;
    	
    	state = STOPPING;
    	try {
	        if (controller != null)
	        {
	        	log.info("Stopping '" + deploymentName + "'");
		        ServicePublisher.removeServicePublishers(controller);
		        controller.stop();
		        controller = null;
		        state = STOPPED;
	        }
    	} catch (Exception e) {
    		throw e;
    	}
    }

    public void destroy() throws Exception
    {
    	if (state == DESTROYED)
    		return;
    	
        log.info("Destroying '" + deploymentName + "'");
        state = DESTROYING;
        
        try {
	        if (deployment != null)
	        {
	            log.info("Undeploying '" + deployment.getSimpleName() + "'");
	            mainDeployer.undeploy(deployment);
	            VirtualFile deploymentFile = deployment.getRoot();
	            if(deploymentFile.exists())
	            {
	                log.info("Deleting '" + deployment.getSimpleName() + "'");
	                if(!deploymentFile.delete())
	                {
	                    log.debug("Failed to delete sub deployment '" + deployment.getName() + "'.");
	                }
	            }
	        }
	
	        LifecycleResourceManager.getSingleton().disassociateDeployment(deploymentName);
	        LifecycleResourceManager.getSingleton().destroyResources();
	        state = DESTROYED;
        } catch (Exception e) {
        	throw e;
        }
    }
    
    private VFSDeployment createVFSDeployment(File war) throws IOException
    {
        VirtualFile webAppVFS = VFS.getRoot(war.toURL());
        return VFSDeploymentFactory.getInstance().createVFSDeployment(webAppVFS);
    }
}
