/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.gateway;

import java.util.ArrayList;
import java.util.Collection;

import org.apache.log4j.Logger;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.HibernateEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.persist.HibernateSessionFactory;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.listeners.lifecycle.AbstractManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;

/**
 * HibernateGatewayListener is a listener which creates a HibernateInterceptor for 
 * the classname/events that need to be listened to.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a>
 * @since Version 4.2
 * 
 */
public class HibernateGatewayListener extends AbstractManagedLifecycle {
	private String m_cfgFile;
	private ConfigTree m_config;
	protected Object _composer;
	protected Class _composerClass;
	protected String _composerName;
	protected String m_targetServiceCategory, m_targetServiceName;
	
	private HibernateInterceptor interceptor ;
	
	protected Collection<EPR> m_targetEprs;
	
	private static final String MESSAGE_FILTER = "messagefilter";
	
	protected final static Logger m_logger = Logger.getLogger(HibernateGatewayListener.class);

	/**
	 * This constructor takes in a configTree element and grabs the hibernate configuration
	 * file name.
	 * 
	 * @param config configtree
	 * @throws ConfigurationException
	 */
	public HibernateGatewayListener(ConfigTree config) throws ConfigurationException {
		super(config);
		m_config = config;
		m_cfgFile = ListenerUtil.getValue(m_config, HibernateEpr.HIBERNATE_CFG_TAG, null);
		
		if (m_cfgFile == null)
			throw new ConfigurationException("No configuration file specified!");
	}

	@Override
	protected void doDestroy() throws ManagedLifecycleException {        
		Configuration cfg = new Configuration();
		cfg.configure(m_cfgFile);
		HibernateSessionFactory.close(cfg);
	}

	@Override
	protected void doInitialise() throws ManagedLifecycleException {
		ArrayList<HibernateEventBean> eventList = new ArrayList<HibernateEventBean>();

		Configuration cfg = new Configuration();
		cfg.configure(m_cfgFile);

		// Need to loop through messagefilters and add them as interceptors
		for (ConfigTree ct : m_config.getChildren(MESSAGE_FILTER)) {
			try {
				String tempEvent = ListenerUtil.getValue(ct, HibernateEpr.EVENT_TAG, null);
				
				if (tempEvent == null)
					throw new ConfigurationException("No event specified.");
				
				String[] events = tempEvent.split(",");
				for (int i = 0; i<events.length; i++) {
					HibernateEventBean heb = new HibernateEventBean();
					heb.setEvent(events[i]);
					heb.setClassname(ListenerUtil.getValue(ct, HibernateEpr.CLASS_NAME_TAG, null));
					
					if (heb.getClassname() == null)
						throw new ConfigurationException("No classname specified.");
					
					eventList.add(heb);	
				}				
			} catch (ConfigurationException ce) {
				m_logger.error("Problem parsing Message Filter event/classname");
			}
		}
		
		try {
			// Grab the session factory and close it - if we have a hot re-deploy,
			// we need to close the SessionFactory so that the old interceptors don't get
			// in the way.
			if (eventList.size() != 0) {
				interceptor = new HibernateInterceptor(m_config, eventList) ;
				cfg.setInterceptor(interceptor);
			}		

			// We're suppressing warnings here - we need to initialize the
			// SessionFactory but don't really need to read from it.
			@SuppressWarnings("unused")
			SessionFactory sf = null;
			HibernateSessionFactory.close(cfg);
			
			sf = HibernateSessionFactory.getInstance(cfg);

		} catch (ConfigurationException ce) {
			throw new ManagedLifecycleException(ce.getMessage());
		}
	}

	@Override
	protected void doStart() throws ManagedLifecycleException {
		if (interceptor != null) {
			interceptor.enable() ;
		}
	}

	@Override
	protected void doStop() throws ManagedLifecycleException {
		if (interceptor != null) {
			interceptor.disable() ;
		}
	}
}
