/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 * (C) 2005-2008, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway.mina;

import org.apache.mina.core.buffer.IoBuffer;
import org.apache.mina.core.session.IdleStatus;
import org.apache.mina.core.service.IoHandlerAdapter;
import org.apache.mina.core.session.IoSession;
import org.apache.mina.transport.socket.SocketSessionConfig;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

/**
 * DefaultMessageHandler recieves messages and packages the bytes sent into
 * a ESB Message object.
 * <p/>
 * The bytes array will be available in the default body location:
 * bytes[] payload = message.getBody().get();
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class DefaultMessageHandler extends IoHandlerAdapter implements MessageHandler
{
    /**
     * ServiceInvoker dispatch to the target service on the ESB.
     */
    private ServiceInvoker invoker;

    /**
     * Sets the service invoker that should be used.
     * 
     * @param invoker The {@link ServiceInvoker} that this message handler should use.
     */
    public final void setServiceInvoker(final ServiceInvoker invoker)
    {
        this.invoker = invoker;
    }

    /**
     * Called when a new connection has been created.
     * 
     * @param session The MINA IoSession.
     */
    public void sessionCreated(final IoSession session)
    {
        if (session.getConfig() instanceof SocketSessionConfig)
        {
            ((SocketSessionConfig) session.getConfig()).setReceiveBufferSize(2048);
        }
        try { 
        	this.sessionIdle(session, IdleStatus.BOTH_IDLE);
        } catch (Exception e) {
        	throw new RuntimeException(e);
        }
    }

    public void exceptionCaught(final IoSession session, final Throwable cause)
    {
        session.close();
    }

    /**
     * Extracts the payload as a bytes array and creates a new ESB Message object populating
     * the EBS Message body with the bytes array.
     * 
     * @param session The MINA IoSession.
     * @param message The MINA message object.
     */
    public void messageReceived(final IoSession session, final Object message) throws Exception
    {
        if (!(message instanceof IoBuffer))
        {
            return;
        }
        
        final IoBuffer byteBuffer = (IoBuffer) message;
        final byte[] bytes = new byte[byteBuffer.limit()];
        byteBuffer.get(bytes);
        
        final Message esbMessage = MessageFactory.getInstance().getMessage();
        esbMessage.getBody().add(bytes);
        invoker.deliverAsync(esbMessage);
    }

}
