/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.message;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.MessageFlowContext;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.*;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;

/**
 * Adapter class for delivering uncomposed (ESB unaware) message payloads to a target service.
 * <p/>
 * Basically wires a {@link org.jboss.soa.esb.client.ServiceInvoker} instance with a {@link MessageComposer}
 * instance.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @see org.jboss.soa.esb.client.ServiceInvoker
 */
public class UncomposedMessageDeliveryAdapter {

    /**
     * Composed message delivery adapter.  Performs the delivery after this class
     * composes the message.
     */
    private org.jboss.soa.esb.client.ServiceInvoker serviceInvoker;
    /**
     * Message composer.
     */
    private MessageComposer composer;
    /**
     * Message flow priority.
     */
    private final Integer messageFlowPriority ;

    /**
     * Public constructor.
     *
     * @param serviceCategory The <b>category name</b> of the Service to which this instance will
     *                        deliver messages.
     * @param serviceName     The <b>name</b> of the Service to which this instance will
     *                        deliver messages.
     * @param composer        The the message composer class for this delivery instance.
     * @throws org.jboss.soa.esb.services.registry.RegistryException
     *          Failed to lookup EPRs for the specified Service.
     */
    public UncomposedMessageDeliveryAdapter(String serviceCategory, String serviceName, MessageComposer composer, final Integer messageFlowPriority) throws MessageDeliverException {
        AssertArgument.isNotNull(serviceCategory, "serviceCategory");
        AssertArgument.isNotNull(serviceName, "serviceName");
        AssertArgument.isNotNull(composer, "composer");
        serviceInvoker = new org.jboss.soa.esb.client.ServiceInvoker(serviceCategory, serviceName);
        this.composer = composer;
        this.messageFlowPriority = messageFlowPriority ;
    }

    /**
     * Synchronously deliver the supplied message to the target service associated with this adapter instance,
     * {@link MessageComposer#decompose(org.jboss.soa.esb.message.Message,Object) decomposing} the reply message payload.
     * <p/>
     * To deliver synchronusly <u>without</u> decomposing the reply, see
     * {@link #deliverSyncWithoutDecomposing(Object,long)}.
     *
     * @param messagePayload Message payload to be packaged, or a channel specific
     *                       container class for the message payload (e.g. a JMS message).
     * @param timeoutMillis  Number of milliseconds before synchronous reply pickup should timeout.
     * @return The reply message payload
     *         ({@link MessageComposer#decompose(org.jboss.soa.esb.message.Message,Object) decomposed}).
     * @throws MessageDeliverException Failed to deliverAsync message, after trying all available EPRs.
     */
    public Object deliverSync(Object messagePayload, long timeoutMillis) throws MessageDeliverException, RegistryException, FaultMessageException {
        // Deliver the message...
        Message message = deliverSyncWithoutDecomposing(messagePayload, timeoutMillis);

        // Decompose the reply message...
        return composer.decompose(message, messagePayload);
    }

    /**
     * Synchronously deliver the supplied message to the target service associated with this adapter instance,
     * without decompsing the synchronous reply message.
     *
     * @param messagePayload Message payload to be packaged, or a channel specific
     *                       container class for the message payload (e.g. a JMS message).
     * @param timeoutMillis  Number of milliseconds before synchronous reply pickup should timeout.
     * @return The reply {@link Message}.  The caller must decompose the message.
     * @throws MessageDeliverException Failed to deliverAsync message, after trying all available EPRs.
     */
    public Message deliverSyncWithoutDecomposing(Object messagePayload, long timeoutMillis) throws MessageDeliverException, RegistryException, FaultMessageException {
        AssertArgument.isNotNull(messagePayload, "messagePayload");
        Message message = composer.compose(messagePayload);

        // Deliver the message...
        MessageFlowContext.setMessageFlowPriority(messageFlowPriority) ;
        try {
            return serviceInvoker.deliverSync(message, timeoutMillis);
        } finally {
            MessageFlowContext.setMessageFlowPriority(null) ;
        }
    }

    /**
     * Asynchronously deliverAsync the supplied message to the target service associated with this adapter instance.
     *
     * @param messagePayload Message payload to be packaged, or a channel specific
     *                       container class for the message payload (e.g. a JMS message).
     * @throws MessageDeliverException Failed to deliverAsync message, after trying all available EPRs.
     */
    public void deliverAsync(Object messagePayload) throws MessageDeliverException, RegistryException {
        AssertArgument.isNotNull(messagePayload, "messagePayload");
        Message message = composer.compose(messagePayload);

        MessageFlowContext.setMessageFlowPriority(messageFlowPriority) ;
        try {
            serviceInvoker.deliverAsync(message);
        } finally {
            MessageFlowContext.setMessageFlowPriority(null) ;
        }
    }

    /**
     * Get the "composed" message delivery adapter instance through which this adpter
     * is delivering message, after composing them.
     *
     * @return Message Delivery Adapter.
     */
    public org.jboss.soa.esb.client.ServiceInvoker getDeliveryAdapter() {
        return serviceInvoker;
    }

    /**
     * Utility factory method for creation of an uncomposed message delivery adapter from a gateway configuration.
     * @param gatewayConfig The gateway configuration.
     * @param defaultComposer The default message composer, used when the configuration doesn't specify a composer.
     * @return The adapter instance.
     * @throws ConfigurationException Invalid gateway configuration.
     */
    public static UncomposedMessageDeliveryAdapter getGatewayDeliveryAdapter(ConfigTree gatewayConfig, MessageComposer defaultComposer) throws ConfigurationException {
        String targetServiceCategory = gatewayConfig.getAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
        String targetServiceName = gatewayConfig.getAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);
        String composerClass = gatewayConfig.getAttribute(ListenerTagNames.GATEWAY_COMPOSER_CLASS_TAG);
        
        final Integer messageFlowPriority = MessageFlowContext.parseMessageFlowPriority(gatewayConfig) ;

        if (targetServiceCategory == null || targetServiceCategory.trim().equals("")) {
            throw new ConfigurationException("Invalid gateway configuration.  '" + ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG + "' not specified.");
        }
        if (targetServiceName == null || targetServiceName.trim().equals("")) {
            throw new ConfigurationException("Invalid gateway configuration.  '" + ListenerTagNames.TARGET_SERVICE_NAME_TAG + "' not specified.");
        }

        try {
            MessageComposer composer;

            if (composerClass != null) {
                composer = MessageComposer.Factory.getInstance(composerClass, gatewayConfig);
            } else {
                composer = defaultComposer;
            }

            composer.setConfiguration(gatewayConfig);

            return new UncomposedMessageDeliveryAdapter(targetServiceCategory, targetServiceName, composer, messageFlowPriority);
        } catch (MessageDeliverException e) {
            throw new ConfigurationException("Remoting Listener configuration failed.", e);
        }
    }
}
