/*
 * JBoss, Home of Professional Open Source
 * Copyright 2011, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.esb.services.security;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Properties;
import java.util.TreeMap;

import org.jboss.logging.Logger;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.util.ClassUtil;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Manage password plugin functionality.
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public final class PasswordPluginManager
{
	private static final Logger logger = Logger.getLogger(PasswordPluginManager.class) ;
	// Default plugins
	private static final PasswordPlugin[] DEFAULT_PLUGINS = { new FilePasswordPlugin() } ;
	
	// The singleton representing the manager.
	private static final PasswordPluginManager MANAGER = new PasswordPluginManager() ;

	// The list of plugins which are configured
	private final PasswordPlugin[] plugins ;
	
	/**
	 * Retrieve a password plugin for the specified file.
	 * @param passwordFile The passwordFile 
	 * @return The PasswordHandler or null if not supported
	 */
	public PasswordHandler passwordHandler(final String passwordFile)
	{
		final URI uri = obtainURI(passwordFile) ;
        
        PasswordHandler passwordHandler = null ;
        if (uri != null)
        {
        	for (PasswordPlugin plugin: plugins)
        	{
        		if (plugin.supportsPasswordFile(uri))
        		{
	        		passwordHandler = plugin.pluginHandler(uri) ;
	        		if (passwordHandler != null)
	        		{
	        			break ;
	        		}
        		}
        	}
        }
		return passwordHandler ;
    }
	
	/**
	 * Retrieve a password plugin for the specified file.
	 * @param passwordFile The passwordFile 
	 * @return The PasswordHandler or null if not supported
	 */
	public boolean isPasswordSupported(final String passwordFile)
	{
		final URI uri = obtainURI(passwordFile) ;
        
        if (uri != null)
        {
        	for (PasswordPlugin plugin: plugins)
        	{
        		if (plugin.supportsPasswordFile(uri))
        		{
        			return true ;
        		}
        	}
        }
		return false ;
    }
	
	{
    	final PropertyManager pm = ModulePropertyManager.getPropertyManager(ModulePropertyManager.SECURITY_MODULE) ;
    	final Properties props = pm.getProperties() ;
		if (props != null)
		{
		    final Enumeration<?> names = props.propertyNames() ;
		    final TreeMap<Integer, String> map = new TreeMap<Integer, String>() ;
		    while (names.hasMoreElements())
		    {
		    	final String name = (String) names.nextElement() ;

		    	if (name.startsWith(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS))
		    	{
		    		final String order = name.substring(Environment.SECURITY_SERVICE_PASSWORD_PLUGINS.length()) ;
				    final Integer value ;
	
				    try
				    {
				    	value = Integer.valueOf(order) ;
				    }
				    catch (NumberFormatException ex)
				    {
				    	throw new RuntimeException("Password plugin name " + order + " is invalid!");
				    }

				    map.put(value, props.getProperty(name)) ;
		    	}
		    }
		    
	    	final ArrayList<PasswordPlugin> pluginList = new ArrayList<PasswordPlugin>(map.size()) ;
	    	
		    if (map.size() > 0)
		    {
		    	final Collection<String> ordered = map.values() ;
		    	final Iterator<String> iter = ordered.iterator() ;

		    	while (iter.hasNext())
		    	{
		    		final String pluginName = iter.next() ;

		    		try
		    		{
		    			final Class<?> c = ClassUtil.forName(pluginName, PasswordPluginManager.class) ;
		    			final PasswordPlugin plugin = (PasswordPlugin) c.newInstance() ;

		    			pluginList.add(plugin);
				    }
				    catch (final ClassNotFoundException cnfe)
				    {
				    	logger.warn("problem loading class " + pluginName, cnfe) ;
				    }
				    catch (final Throwable th)
				    {
				    	logger.warn("problem during load " + pluginName, th) ;
				    }
		    	}
		    	if (pluginList.size() > 0)
		    	{
		    		plugins = pluginList.toArray(new PasswordPlugin[0]) ;
		    	}
		    	else
		    	{
		    		plugins = DEFAULT_PLUGINS ;
		    	}
		    }
		    else
		    {
		    	plugins = DEFAULT_PLUGINS ;
		    }
		}
		else
		{
	    	plugins = DEFAULT_PLUGINS ;
		}
    }
    
    private static URI obtainURI(final String passwordFile)
    {
        if (passwordFile == null || "".equals(passwordFile))
            return null ;
        
        URI uri = null ;
        try
		{
        	uri = new URI(passwordFile) ;
		}
		catch (final URISyntaxException ignore) {}

        if ((uri == null) || !uri.isAbsolute())
        {
	        final File pwFile = new File(passwordFile);
			if (pwFile.exists() && !pwFile.isDirectory())
			{
				uri = pwFile.toURI() ;
			}
        }
        return uri ;
    }
    
    public static PasswordPluginManager getManager()
    {
    	return MANAGER ;
    }
}
