/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import java.security.Principal;
import java.util.List;

import javax.security.auth.Subject;
import javax.security.auth.login.LoginException;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;

/**
 * This interface represents a Security service.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * @since 4.4
 *
 */
public interface SecurityService
{
	/**
	 * Constant intended for usage as a key in different types of maps.
	 */
	String CONTEXT = "org.jboss.soa.esb.services.security.context";

	String AUTH_REQUEST = "org.jboss.soa.esb.services.security.authRequest";

	/**
	 * Configure the security serivce for usage. This should take care of
	 * reading any needed configurations required by the security system.
	 *
	 * @throws ConfigurationException
	 */
	void configure() throws ConfigurationException;

	/**
	 * Authenticates a subject.
	 *
	 * @param securityContext - the security context to authenticate
	 * @throws LoginException - if the authentication was not sucessful
	 */
	void authenticate(final SecurityConfig securityConfig, final SecurityContext securityContext, final AuthenticationRequest authRequest) throws SecurityServiceException;

	/**
	 * Check if the the SecurityContext has a principal that is a member of one
	 * of the passed in rolesAllowed list.
	 *
	 * @param rolesAllowed - list or allowed roles
	 * @param securityContext - the security context
	 * @return true - if the SecurityContext has a principal that is a member of one the roles.
	 */
	boolean checkRolesAllowed(final List<String> rolesAllowed, SecurityContext securityContext);

	/**
	 * Determines if the subject contains the passed in role in it's
	 * set of Principals.
	 *
	 * @param subject	- the Subject instance belonging to the caller.
	 * @param context	- the principal instance representing the role to check for.
	 * @return true		- if the caller has the role of
	 */
	boolean isCallerInRole(final Subject subject, final Principal role);

	/**
	 * Logout and clean up of any security state.
	 *
	 * @param securityContext
	 * @throws LoginException
	 */
	void logout(final SecurityConfig securityConfig);

	/**
	 * Refresh the security configuration
	 */
	void refreshSecurityConfig();
}
