/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth;

import java.io.Serializable;
import java.security.Principal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.jboss.soa.esb.services.security.principals.User;

/**
 * AuthenticationRequestImpl is indended to transport security related information
 * needed to authenticate a caller.
 * </p>
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class AuthenticationRequestImpl implements Serializable, AuthenticationRequest
{
	private static final long serialVersionUID = 1L;

	private static transient ThreadLocal<byte[]> encryptedAuthRequest = new ThreadLocal<byte[]>();

	private Principal principal;
	private Set<?> credentials;
	private Map<String,?> properties;

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.services.security.auth.AuthenticationRequest#getCredentials()
	 */
	public Set<?> getCredentials()
	{
		return credentials;
	}

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.services.security.auth.AuthenticationRequest#getProperties()
	 */
	public Map<String, ?> getProperties()
	{
		return properties;
	}

	/* (non-Javadoc)
	 * @see org.jboss.soa.esb.services.security.auth.AuthenticationRequest#getPrincipal()
	 */
	public Principal getPrincipal()
	{
		return principal;
	}

	public static byte[] getEncryptedAuthRequest()
    {
        return encryptedAuthRequest.get();
    }

    public static void setEncryptedAuthRequest(byte[] encryptedAuthRequest)
    {
        AuthenticationRequestImpl.encryptedAuthRequest.set(encryptedAuthRequest);
    }

    public static class Builder
	{
    	private Principal principal;
    	private Set<Object> credentials = new HashSet<Object>();
    	private Map<String,Object> properties = new HashMap<String,Object>();

    	public Builder() {}
    	
    	public Builder(final Principal principal)
    	{
    		this.principal = principal;
    	}

    	public Builder(final Principal principal, final Set<Object> credentials)
    	{
    	    this(principal);
    		if ( credentials != null )
    		{
        		this.credentials.addAll(credentials);
    		}
    	}

    	public Builder property( final String name, final Object value)
    	{
    		properties.put(name, value);
    		return this;
    	}

    	public Builder username( final String username )
    	{
    		principal = new User( username );
    		return this;
    	}

    	public Builder password( final char[] password )
    	{
    		credentials.add(password);
    		return this;
    	}

    	public AuthenticationRequest build()
    	{
    		return new AuthenticationRequestImpl(this);
    	}
	}

	private AuthenticationRequestImpl(final Builder builder)
	{
		principal = builder.principal;
		credentials = builder.credentials;
		properties = builder.properties;
	}

}
