/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.io.Serializable;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.ErrorListener;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;

import org.apache.log4j.Appender;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Logger;
import org.apache.log4j.TTCCLayout;
import org.jboss.internal.soa.esb.message.format.xml.MessageImpl;
import org.jboss.internal.soa.esb.message.format.xml.XMLUtil;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.message.Message;
import org.xml.sax.SAXException;

public class Util
{
    private Util()
    {
    }

    public static void dumpSysProps (PrintStream p_OS)
    {
	String[] sa = new String[ModulePropertyManager.getPropertyManager(
		ModulePropertyManager.TRANSPORTS_MODULE).getProperties().size()];
	ModulePropertyManager.getPropertyManager(
		ModulePropertyManager.TRANSPORTS_MODULE).getProperties()
		.keySet().toArray(sa);
	Arrays.sort(sa);
	for (String sCurr : sa)
	    p_OS.println(sCurr
		    + "="
		    + ModulePropertyManager.getPropertyManager(
			    ModulePropertyManager.TRANSPORTS_MODULE)
			    .getProperty(sCurr));
    } // __________________________________

    public static boolean isNullString (String p_s)
    {
	return (null == p_s) ? true : p_s.trim().length() < 1;
    } // __________________________________

    public static boolean isLong (String p_s)
    {
	if (isNullString(p_s))
	    return false;

	try
	{
	    Long.parseLong(p_s);
	    return true;
	}
	catch (Exception e)
	{
	    return false;
	}
    } // __________________________________

    public static boolean isPositiveLong (String p_s)
    {
	if (isNullString(p_s))
	    return false;

	try
	{
	    long lBk = Long.parseLong(p_s);
	    return lBk > 0;
	}
	catch (Exception e)
	{
	    return false;
	}
    } // __________________________________

    public static int parseInt (String s)
    {
	if (s == null)
	    return 0;

	String sVal = s.trim();

	if (sVal.length() < 1)
	    return 0;
	else
	{
	    try
	    {
		return Integer.parseInt(s);
	    }
	    catch (NumberFormatException ex)
	    {
		return 0;
	    }
	}
    } // __________________________________

    public static long parseLong (String s)
    {
	if (s == null)
	    return 0;

	String sVal = s.trim();

	if (sVal.length() < 1)
	    return 0;
	else
	{
	    try
	    {
		return Long.parseLong(s);
	    }
	    catch (NumberFormatException ex)
	    {
		return 0;
	    }
	}
    } // __________________________________

    public static Logger getDefaultLogger (Class p_oCls)
    {
	Logger oRet = Logger.getLogger(p_oCls.getName());
	Appender oApp = new ConsoleAppender(new TTCCLayout("ISO8601"));
	oRet.addAppender(oApp);
	return oRet;
    } // __________________________________

    // ParserConfigurationException is no longer thrown but,
    // unfortunately, it is in the signature of the method!
    public static Serializable serialize (Message message)
	    throws ParserConfigurationException, IOException
    {
	if (message.getType().equals(
		org.jboss.soa.esb.message.format.MessageType.JAVA_SERIALIZED))
	    return (Serializable) message;

	try
	{
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        final XMLStreamWriter out = XMLHelper.getXMLStreamWriter(baos) ;
        final String origURI = StreamHelper.writeStartElement(out, XMLUtil.ESB_QNAME_ENVELOPE) ;
            ((MessageImpl) message).writeContent(out) ;
        StreamHelper.writeEndElement(out, XMLUtil.ESB_QNAME_ENVELOPE.getPrefix(), origURI) ;
        out.flush();
        int size = baos.toByteArray().length;
        message.getProperties().setProperty(Environment.MESSAGE_BYTE_SIZE, "" + size);
        message.getProperties().setProperty(Environment.MESSAGE_TIME_PROCESSED, "0");
        return baos.toString();
	}
	catch (final XMLStreamException xmlse)
	{
            final IOException ioe = new IOException("Util.serialize caught XMLStreamException:") ;
            ioe.initCause(xmlse) ;
            throw ioe ;
	}
    }// ________________________________

    // ParserConfigurationException and SAXException are no longer thrown
    // but, unfortunately, they are in the signature of the method!
    public static Message deserialize (Serializable serial)
	    throws ParserConfigurationException, SAXException, IOException
    {
	if (serial instanceof Message) // MessageType.JAVA_SERIALIZED
	    return (Message) serial;

	try
	{
        // MessageType.JBOSS_XML
		int size = ((String)serial).getBytes().length;
	    final StringReader reader = new StringReader((String)serial);
	    final XMLStreamReader in = XMLHelper.getXMLStreamReader(reader) ;
	    StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_ENVELOPE) ;
	    Message mess = new MessageImpl(in);
	    mess.getProperties().setProperty(Environment.MESSAGE_BYTE_SIZE, "" + size);
	    return mess;
	}
	catch (XMLStreamException xmlse)
	{
           final IOException ioe = new IOException("Util.deserialize caught XMLStreamException") ;
           ioe.initCause(xmlse) ;
           throw ioe ;
	}
    } // ________________________________

    public static List<KeyValuePair> propertiesFromSelector (String selector)
    {
	// No problem if selector is null - everything in queue will be returned
	List<KeyValuePair> oRet = new ArrayList<KeyValuePair>();

	if (!Util.isNullString(selector))
	{
	    for (String sCurr : selector.trim().split("AND"))
	    {
		String[] sa = sCurr.split("=");
		String attr = null;
		String value = null;
		boolean fail = false;
		
		if (sa.length == 2)
		{
			attr = sa[0].trim();
			value= sa[1].trim();
			if (value.charAt(0) != '\''
				|| value.charAt(value.length() - 1) != '\'')
			{
				fail = true;
			}
		}
		else
			fail = true;
		
		if (fail)
		{
		    throw new IllegalArgumentException(
			    "Illegal message selector syntax <"
				    + selector
				    + ">.  Must be 2 tokens seperated by an '=' character, and the token after the '=' character must be enclosed in single quotes.");
		}
		
		// attr/value will be non-null by here.
		KeyValuePair oNew = new KeyValuePair(attr, value.substring(0,
			-1 + value.length()).substring(1));
		oRet.add(oNew);
	    }
	}

	return oRet;
    } // ________________________________

    public static String getStamp ()
    {
	return s_oTS.format(new java.util.Date(System.currentTimeMillis()));
    }

    /**
         * Attempt to format the output.
         * 
         * @param output
         *                Output to be formatted.
         * @return Output.
         */

    public static String format (String output)
    {
	/*
	 * If it's not XML, then don't try to deal with it as if it were.
	 */
	
	if (!output.startsWith("<?xml"))
	    return output;
	
	StreamSource source = new StreamSource(new ByteArrayInputStream(output
		.getBytes()));
	ByteArrayOutputStream outStream = new ByteArrayOutputStream();

	try
	{
	    Transformer transformer;

	    try
	    {
		factory.setAttribute("indent-number", Integer.valueOf(4));
	    }
	    catch (Exception e)
	    {
		// Ignore... Xalan may throw on this!!
		// We handle Xalan indentation below (yeuckkk) ...
	    }
	    transformer = factory.newTransformer();
	    transformer.setOutputProperty(OutputKeys.INDENT, "yes");
	    // There's a bug in Java 5 re this code (formatting).
	    // See
                // http://forum.java.sun.com/thread.jspa?threadID=562510&start=0
	    // and it explains the
	    // whys of the following code.
	    transformer.setOutputProperty(
		    "{http://xml.apache.org/xalan}indent-amount", "4");
	    transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION,
		    "yes");
	    transformer.setErrorListener(new NullErrorListener());
	    transformer.transform(source, new StreamResult(outStream));

	    return outStream.toString();
	}
	catch (Exception e)
	{
	    return output;
	}
    }

    private static TransformerFactory factory = TransformerFactory
	    .newInstance();

    private static class NullErrorListener implements ErrorListener
    {
	public void warning (TransformerException exception)
		throws TransformerException
	{
	}

	public void error (TransformerException exception)
		throws TransformerException
	{
	}

	public void fatalError (TransformerException exception)
		throws TransformerException
	{
	}
    }

    private static final SimpleDateFormat s_oTS = new SimpleDateFormat(
	    "yyyy/MM/dd hh:mm:ss.SSS");
} // ____________________________________________________________________________
