/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.cbr.xpath;

import junit.framework.TestCase;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.actions.cbr.XPathRouter;
import org.jboss.soa.esb.actions.cbr.RoutingRule;
import org.xml.sax.SAXException;

import javax.xml.namespace.NamespaceContext;
import java.io.IOException;
import java.io.File;
import java.io.ByteArrayInputStream;
import java.util.Map;
import java.util.List;
import java.util.Arrays;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class XPathRouterUnitTest extends TestCase {

    public void test_rulefile_load_classpath() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        test("1");
    }

    public void test_rulefile_load_filesys() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        String cwd = new File(".").getAbsolutePath().replace("\\", "/");
        if(!cwd.endsWith("product/.")) {
            throw new IllegalStateException("Invalid working directory for test.  Must be the 'product/rosetta/tests' dir.  Current working directory is '" + cwd + "'.");
        }

        test("2");
    }

    public void test_inlineRules() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        test("3");
    }

    public void test_namespaces_undefined() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        try {
            test("4");
            fail("Expected MessageRouterException");
        } catch (MessageRouterException e) {
            assertEquals("Error compiling XPath expression '/a:order/b:header[@cat='blue']'.", e.getMessage());
        }
    }

    public void test_namespaces_defined() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        NamespaceContext nsContext = test("5");

        assertEquals("http://a", nsContext.getNamespaceURI("a"));
        assertEquals("http://b", nsContext.getNamespaceURI("b"));
    }

    private NamespaceContext test(String configName) throws MessageRouterException, IOException, SAXException, ConfigurationException {
        ESBConfigUtil esbConfig = new ESBConfigUtil(getClass().getResourceAsStream("config.xml"));
        ConfigTree configTree = esbConfig.getActionConfig("null-listener", configName);
        XPathRouter router = new XPathRouter();

        router.setConfigTree(configTree);

        Map<String, RoutingRule> map = router.getRoutingMap();
        assertEquals(3, map.size());
        assertNotNull(map.get("scat-red"));
        assertNotNull(map.get("scat-green"));
        assertNotNull(map.get("scat-blue"));

        return router.getNamespaceContext();
    }

    public static final String GREEN_MESSAGE = "<order xmlns='http://a' xmlns:bbbb='http://b'><bbbb:header cat='green'/></order>";

    public void test_routing_String() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(GREEN_MESSAGE);
        test_routing(message, null);
    }

    public void test_routing_bytes() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(GREEN_MESSAGE.getBytes());
        test_routing(message, null);
    }

    public void test_routing_dom() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add(YADOMUtil.parseStream(new ByteArrayInputStream(GREEN_MESSAGE.getBytes()), true, true, true));
        test_routing(message, null);
    }

    public void test_routing_ObjectList() throws IOException, SAXException, ConfigurationException, MessageRouterException {
        Message message = MessageFactory.getInstance().getMessage();

        test_routing(message, Arrays.asList(GREEN_MESSAGE));
    }

    public void test_routing(Message message, List objectLists) throws IOException, SAXException, ConfigurationException, MessageRouterException {
        ESBConfigUtil esbConfig = new ESBConfigUtil(getClass().getResourceAsStream("config.xml"));
        ConfigTree configTree = esbConfig.getActionConfig("null-listener", "5");
        XPathRouter router = new XPathRouter();

        router.setConfigTree(configTree);

        List<String> destinations = router.route(null, false, message, objectLists);
        assertEquals("[scat-green]", destinations.toString());
    }
}
