/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

package org.jboss.soa.esb.actions.routing.http;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.io.IOException;
import java.net.URL;

import junit.framework.JUnit4TestAdapter;

import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpMethodRetryHandler;
import org.apache.commons.httpclient.params.HttpMethodParams;
import org.jboss.soa.esb.Configurable;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

public class HttpMethodFactoryUnitTest
{
    @Test
    public void testDefaultRetryHandlerPOST()
        throws Exception
    {
        testDefaultRetryHandler("POST") ;
    }

    @Test
    public void testDefaultRetryHandlerGET()
        throws Exception
    {
        testDefaultRetryHandler("GET") ;
    }

    private void testDefaultRetryHandler(final String method)
        throws Exception
    {
        final HttpMethodBase httpMethodBase = executeTest(method, null) ;
        final HttpMethodParams params = httpMethodBase.getParams() ;
        final Object handler = params.getParameter(HttpMethodParams.RETRY_HANDLER) ;
        assertNotNull("retry handler", handler) ;
        assertEquals("handler class", AbstractHttpMethodFactory.HttpMethodRetryHandlerWrapper.class, handler.getClass()) ;
    }

    @Test
    public void testRetryHandlerPOST()
        throws Exception
    {
        testRetryHandler("POST") ;
    }

    @Test
    public void testRetryHandlerGET()
        throws Exception
    {
        testRetryHandler("GET") ;
    }

    private void testRetryHandler(final String method)
        throws Exception
    {
        final Class<?> handlerClass = RetryHandler.class ;
        final HttpMethodBase httpMethodBase = executeTest(method, handlerClass.getName()) ;
        final HttpMethodParams params = httpMethodBase.getParams() ;
        final Object handler = params.getParameter(HttpMethodParams.RETRY_HANDLER) ;
        assertNotNull("retry handler", handler) ;
        assertEquals("handler class", handlerClass, handler.getClass()) ;
    }

    @Test
    public void testConfigurableRetryHandlerPOST()
        throws Exception
    {
        testConfigurableRetryHandler("POST") ;
    }

    @Test
    public void testConfigurableRetryHandlerGET()
        throws Exception
    {
        testConfigurableRetryHandler("GET") ;
    }

    private void testConfigurableRetryHandler(final String method)
        throws Exception
    {
        final Class<ConfigurableRetryHandler> handlerClass = ConfigurableRetryHandler.class ;
        final HttpMethodBase httpMethodBase = executeTest(method, handlerClass.getName()) ;
        final HttpMethodParams params = httpMethodBase.getParams() ;
        final Object handler = params.getParameter(HttpMethodParams.RETRY_HANDLER) ;
        assertNotNull("retry handler", handler) ;
        assertEquals("handler class", handlerClass, handler.getClass()) ;
        final ConfigurableRetryHandler configurableRetryHandler = handlerClass.cast(handler) ;
        assertNotNull("config", configurableRetryHandler.getConfig()) ;
    }

    private HttpMethodBase executeTest(final String method, final String handlerName)
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getBody().add("test") ;
        final ConfigTree config = new ConfigTree("test") ;
        if (handlerName != null)
        {
            config.setAttribute(AbstractHttpMethodFactory.RETRY_HANDLER, handlerName) ;
        }
        final HttpMethodFactory factory = HttpMethodFactory.Factory.getInstance(method, config, new URL("http://dummy")) ;
        return factory.getInstance(message) ;
    }

    public static final class RetryHandler implements HttpMethodRetryHandler
    {
        public boolean retryMethod(final HttpMethod method, final IOException exception, final int executionCount)
        {
            return false;
        }
    }

    public static final class ConfigurableRetryHandler implements HttpMethodRetryHandler, Configurable
    {
        private ConfigTree config ;

        public boolean retryMethod(final HttpMethod method, final IOException exception, final int executionCount)
        {
            return false;
        }

        public void setConfiguration(final ConfigTree config)
            throws ConfigurationException
        {
            this.config = config ;
        }

        public ConfigTree getConfig()
        {
            return config ;
        }
    }

    public static junit.framework.Test suite() {
        return new JUnit4TestAdapter(HttpMethodFactoryUnitTest.class);
    }
}
