/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners;

import junit.framework.TestCase;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.mock.MockAction;
import org.jboss.soa.esb.parameters.ParamRepositoryException;
import org.jboss.soa.esb.testutils.AbstractTestRunner;
import org.jboss.soa.esb.testutils.ESBConfigUtil;
import org.xml.sax.SAXException;

import java.io.IOException;
import java.util.List;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class InVMListenerUnitTest extends TestCase {

    protected void setUp() throws Exception {
        System.setProperty(Environment.DEFAULT_INVM_SCOPE, "GLOBAL");
        MockAction.exception = null;
    }

    protected void tearDown() throws Exception {
        System.setProperty(Environment.DEFAULT_INVM_SCOPE, "NONE");
        MockAction.exception = null;
    }

    public void test_async() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("ServiceCat", "ServiceName");
                Message message = MessageFactory.getInstance().getMessage();

                message.getBody().add("Hi there!");
                invoker.deliverAsync(message);

                waitForMockSet(message);
                assertTrue("Message equality", checkMessageEquality(message, MockAction.message));
            }
        }.setServiceConfig("in-listener-config-01.xml");

        testRunner.run();
    }

    public void test_async_lockstep() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                // Test that we can invoke them....
                invokeService("ServiceCat", "Service1");
                invokeService("ServiceCat", "Service2");

                // Test that the EPRs are as we expected...
                List<EPR> eprs = RegistryUtil.getEprs("ServiceCat", "Service1");
                assertEquals(1, eprs.size());
                assertEquals(false, ((InVMEpr)eprs.get(0)).getLockstep());
                assertEquals(10000, ((InVMEpr)eprs.get(0)).getLockstepWaitTime());

                eprs = RegistryUtil.getEprs("ServiceCat", "Service2");
                assertEquals(1, eprs.size());
                assertEquals(true, ((InVMEpr)eprs.get(0)).getLockstep());
                assertEquals(4000, ((InVMEpr)eprs.get(0)).getLockstepWaitTime());
            }
        }.setServiceConfig("in-listener-config-04.xml");

        testRunner.run();
    }

    public void test_sync_noerror() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("ServiceCat", "ServiceName");
                Message message = MessageFactory.getInstance().getMessage();

                message.getBody().add("Hi there!");
                Message response = invoker.deliverSync(message, 2000);

                assertTrue("Message equality", checkMessageEquality(message, MockAction.message));
                assertTrue("Message equality", checkMessageEquality(message, response));
            }
        }.setServiceConfig("in-listener-config-01.xml");

        testRunner.run();
    }

    public void test_sync_error() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("ServiceCat", "ServiceName");
                Message message = MessageFactory.getInstance().getMessage();

                message.getBody().add("Hi there!");
                MockAction.exception = new ActionProcessingException("invm_sync_error");
                try {
                    invoker.deliverSync(message, 2000);
                    fail("Expected FaultMessageException.");
                } catch(FaultMessageException e) {
                    assertEquals("org.jboss.soa.esb.actions.ActionProcessingException: invm_sync_error", e.getMessage());
                }
            }
        }.setServiceConfig("in-listener-config-01.xml");

        testRunner.run();
    }

    public void test_JBESB_1865() throws Exception {
        ESBConfigUtil configUtil = new ESBConfigUtil(getClass().getResourceAsStream("in-listener-config-JBESB-1865.xml"));

        String maxThreads = configUtil.getListenerConfig("InVM-0").getAttribute(ListenerTagNames.MAX_THREADS_TAG);
        assertEquals("10", maxThreads);
    }

    public void test_passByValue() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("ServiceCat", "ServiceName");
                Message message = MessageFactory.getInstance().getMessage();

                message.getBody().add("This message was passed by value!");
                invoker.deliverSync(message, 2000);

                assertTrue(message != MockAction.message);
                assertEquals("This message was passed by value!", MockAction.message.getBody().get());
            }
        }.setServiceConfig("in-listener-config-05.xml");

        testRunner.run();
    }

    public void test_sync_multithreaded() throws Exception {
        AbstractTestRunner testRunner = new AbstractTestRunner() {
            public void test() throws Exception {
                ServiceInvoker invoker = new ServiceInvoker("ServiceCat", "ServiceName");
                ClientInvokerThread[] clients = new ClientInvokerThread[10];

                // Create the clients...
                for (int i = 0; i < clients.length; i++) {
                    clients[i] = new ClientInvokerThread(invoker);
                }

                // Start the clients...
                for (int i = 0; i < clients.length; i++) {
                    clients[i].start();
                }

                // Wait for the clients to be "done"...
                boolean allNotDone = true;
                while(allNotDone) {
                    sleep(100);
                    allNotDone = false;
                    for (int i = 0; i < clients.length; i++) {
                        if(!clients[i].done) {
                            allNotDone = true;
                            break;
                        }
                    }
                }

                // Check were there any assertion failures...
                for (int i = 0; i < clients.length; i++) {
                    if(clients[i].thrown != null) {
                        fail("Thread " + i + " was in error: " + clients[i].thrown.getMessage());
                    }
                }
            }
        }.setServiceConfig("in-listener-config-03.xml");

        testRunner.run();
    }

    public void test_none_scoped() throws ManagedLifecycleException, SAXException, ParamRepositoryException, MessageDeliverException, IOException {
        try {
            new ESBConfigUtil(getClass().getResourceAsStream("in-listener-config-02.xml"));
            fail("Expected ConfigurationException");
        } catch (ConfigurationException e) {
            assertEquals("Service 'ServiceCat:ServiceName' does not define a Message-Aware (is-gateway='false') nor InVm Listener.", e.getMessage());
        }
    }

    private class ClientInvokerThread extends Thread {

        private ServiceInvoker invoker = null;

        private boolean done = false;

        private Throwable thrown;

        private ClientInvokerThread(ServiceInvoker invoker) {
            this.invoker = invoker;
        }

        public void run() {
            try {
                for(int i = 0; i < 50; i++) {
                    Message message = MessageFactory.getInstance().getMessage();
                    Message response = null;

                    try {
                        response = invoker.deliverSync(message, 5000);
                    } catch (Exception e) {
                        fail("Error delivering message: " + e.getMessage());
                    }
                    assertTrue("Message equality", checkMessageEquality(message, response));
                    assertEquals("Tom Fennelly", response.getBody().get());

                    InVMListenerUnitTest.sleep(10);
                }
            } catch(Throwable t) {
                this.thrown = t;
            } finally {
                done = true;
            }
        }

    }

    private void invokeService(String cat, String name) throws MessageDeliverException {
        ServiceInvoker invoker = new ServiceInvoker(cat, name);

        Message message = MessageFactory.getInstance().getMessage();

        message.getBody().add("Hi there!");
        invoker.deliverAsync(message);

        waitForMockSet(message);
        assertTrue("Message equality", checkMessageEquality(message, MockAction.message));
    }

    private void waitForMockSet(Message message) {
        long start = System.currentTimeMillis();

        while(System.currentTimeMillis() - start < 5000) {
            if(checkMessageEquality(message, MockAction.message)) {
                return;
            }
            sleep(50);
        }
    }

    private static void sleep(long millis) {
        try {
            Thread.sleep(millis);
        } catch (InterruptedException e) {
            fail("Unexpected InterruptedException exception.");
        }
    }
    
    private static boolean checkMessageEquality(final Message expected, final Message actual) {
        return (actual != null) && (expected.getAttachment().equals(actual.getAttachment())) &&
            (expected.getBody().equals(actual.getBody())) &&
            (expected.getFault().equals(actual.getFault())) &&
            (expected.getProperties().equals(actual.getProperties())) &&
            (expected.getType().equals(actual.getType())) ;
    }
}