/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc.util;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for {@link VfsUtil}.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class VfsUtilUnitTest
{
    @BeforeClass
    public static void initVFS()
    {
        VFS.init();
    }

    @Test
    public void createInMemoryFs() throws IOException
    {
        final URL rootUrl = new URL("vfsmemory", "VfsUtilUnitTest", "");
        final  VirtualFile root = VfsUtil.createInMemoryFs(rootUrl);
        assertNotNull(root);

        final VirtualFile root2 = VfsUtil.createInMemoryFs(rootUrl);
        assertEquals(root, root2);
    }

    @Test
    public void addFile() throws IOException
    {
        final URL rootUrl = new URL("vfsmemory", "VfsUtilUnitTest", "");
        final VirtualFile rootFs = VfsUtil.createInMemoryFs(rootUrl);

        VfsUtil.addFile(rootUrl, "test.xml", "bajja".getBytes());
        final VirtualFile testFile = rootFs.getChild("test.xml");
        assertNotNull(testFile);
    }

    @Test
    public void createUrl() throws MalformedURLException
    {
        URL rootUrl = VfsUtil.createInMemUrl("vsftest");
        assertEquals("vfsmemory://vsftest", rootUrl.toExternalForm());
    }

    @Test
    public void createUrlPath() throws MalformedURLException
    {
        URL rootUrl = VfsUtil.createInMemUrl("vsftest", "WEB-INF");
        assertEquals("vfsmemory://vsftest/WEB-INF", rootUrl.toExternalForm());
    }

    @Test
    public void createUrlPathUrl() throws IOException
    {
        URL rootUrl = VfsUtil.createInMemUrl("vsftest");
        URL pathUrl = VfsUtil.createInMemUrl(rootUrl, "WEB-INF/classes");
        assertEquals("vfsmemory://vsftest/WEB-INF/classes", pathUrl.toExternalForm());
    }

    @Test
    public void addFilesInFromInMemFs() throws IOException, URISyntaxException
    {
        URL sourceUrl = VfsUtil.createInMemUrl("vsftest");
        VfsUtil.createInMemoryFs(sourceUrl);
        VirtualFile file1 = VfsUtil.addFile(sourceUrl, "test", "dummy".getBytes());
        VirtualFile file2 = VfsUtil.addFile(sourceUrl, "test2", "dummy".getBytes());

        URL destUrl = VfsUtil.createInMemUrl("vsftest2");
        VirtualFile destDir = VfsUtil.createInMemoryFs(destUrl);

        VfsUtil.addFiles(Arrays.asList(file1, file2), destUrl, null);

        assertNotNull(destDir.getChild("test"));
        assertNotNull(destDir.getChild("test2"));
    }

    @Test
    public void addSchemasInFromInMemFs() throws IOException, URISyntaxException
    {
        URL sourceUrl = VfsUtil.createInMemUrl("vsftest");
        VfsUtil.createInMemoryFs(sourceUrl);
        VirtualFile file1 = VfsUtil.addFile(sourceUrl, "/rootPath/somepath/types/test.xsd", "dummy".getBytes());
        VirtualFile file2 = VfsUtil.addFile(sourceUrl, "/rootPath/somepath/types/test2.xsd", "dummy".getBytes());

        URL destUrl = VfsUtil.createInMemUrl("vsftest2");
        VirtualFile destDir = VfsUtil.createInMemoryFs(destUrl);

        VfsUtil.addFiles(Arrays.asList(file1, file2), destUrl, "somepath", "WEB-INF/wsdl");

        assertNotNull(destDir.getChild("WEB-INF/wsdl/types/test.xsd"));
        assertNotNull(destDir.getChild("WEB-INF/wsdl/types/test2.xsd"));
    }
    
    @Test
    public void addSchemas() throws IOException, URISyntaxException
    {
        URL sourceUrl = VfsUtil.createInMemUrl("vsftest");
        VirtualFile file1 = VfsUtil.addFile(sourceUrl, "/rootPath/schema.jar/types/test.xsd", "dummy".getBytes());
        VirtualFile file2 = VfsUtil.addFile(sourceUrl, "/rootPath/schema.jar/types/test2.xsd", "dummy".getBytes());
        HashMap<VirtualFile, String> map = new HashMap<VirtualFile, String>();
        
        // specify that file1 should be given a different path
        map.put(file1, "newpath/test1.xsd");
        // specify that file2 should be given a different path
        map.put(file2, "newpath/test2.xsd");
        
        URL destUrl = VfsUtil.createInMemUrl("vsftest3");
        VirtualFile destDir = VfsUtil.createInMemoryFs(destUrl);
        VfsUtil.addFiles(map, destUrl, "WEB-INF/wsdl/" + destDir.getName());
        
        assertNotNull(destDir.getChild("WEB-INF/wsdl/newpath/test1.xsd"));
        assertNotNull(destDir.getChild("WEB-INF/wsdl/newpath/test2.xsd"));
    }
    
    @Test
    public void addFilesInFromFileFs() throws IOException, URISyntaxException
    {
        File workDir = null;
        try
        {
            // Create a working directory in the OS tmp dir.
            File tmpFile = File.createTempFile("test", "junit");
            tmpFile.deleteOnExit();
            File tmpDir = new File(tmpFile.getParent());
            workDir = new File(tmpDir, "VfsTest");
            workDir.mkdir();

            // Add two files to the working directory.
            File file1 = new File(workDir, "test1.txt");
            file1.createNewFile();
            File file2 = new File(workDir, "test2.txt");
            file2.createNewFile();

            // Create a new virtual root using the workDir.
            VirtualFile root = VFS.createNewRoot(workDir.toURI());
            List<VirtualFile> files = root.getChildren();
            assertEquals(2, files.size());

            URL destUrl = VfsUtil.createInMemUrl("vsftest2");
            VirtualFile destDir = VfsUtil.createInMemoryFs(destUrl);

            VfsUtil.addFiles(files, destUrl, null);
            assertNotNull(destDir.getChild("test"));
            assertNotNull("test1.txt", destDir.getChild("test").getPathName());
            assertNotNull(destDir.getChild("test2"));
            assertNotNull("test2.txt", destDir.getChild("test2").getPathName());
        }
        finally
        {
            if (workDir != null)
            {
                File[] listFiles = workDir.listFiles();
                for (File file : listFiles)
                {
                    file.delete();
                }
                workDir.delete();
            }
        }
    }

    @Test
    public void subpath()
    {
        String actualPath = "Test.esb/types/custom-request-type.xsd";
        String expectedPath = "/types/custom-request-type.xsd";
        assertEquals(expectedPath, VfsUtil.subpath(actualPath, "Test.esb"));
        assertEquals(null, VfsUtil.subpath(null, "Test.esb"));
        assertEquals(actualPath, VfsUtil.subpath(actualPath, null));
        assertEquals(actualPath, VfsUtil.subpath(actualPath, "dummy"));
    }

    @Test
    public void subpathWithSuffix()
    {
        String actualPath = "Test.esb/types/custom-request-type.xsd";
        String expectedPath = "WEB-INF/wsdl/types/custom-request-type.xsd";

        assertEquals(expectedPath, VfsUtil.subpath(actualPath, "Test.esb", "WEB-INF/wsdl"));
    }

    @Test
    public void subpathWithoutPrefix()
    {
        String actualPath = "types/custom-request-type.xsd";
        String expectedPath = "WEB-INF/wsdl/types/custom-request-type.xsd";

        assertEquals(expectedPath, VfsUtil.subpath(actualPath, "Test.esb", "WEB-INF/wsdl"));
    }
}
