/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.persistence.actions;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.InputStream;
import java.net.URI;
import java.sql.Connection;
import java.sql.Statement;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.xml.DOMConfigurator;
import org.jboss.internal.soa.esb.persistence.format.MessageStoreFactory;
import org.jboss.soa.esb.actions.MessagePersister;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.persistence.manager.ConnectionManager;
import org.jboss.soa.esb.persistence.manager.ConnectionManagerFactory;
import org.jboss.soa.esb.services.persistence.MessageStore;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.jboss.soa.esb.testutils.TestEnvironmentUtil;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Testing the MessagePersister.
 * 
 * @author <a href="mailto:kurt.stam@redhat.com">Kurt Stam</a>
 * 
 */
public class MessagePersisterUnitTest
{
	private static Object server ;
	
	/**
	 * Testing the MessagePersister.
	 */
	@Test
	public void sendMessage()
	{
		try
		{
            Date now = new Date();
            String body = "<message>First Message " + now + " </message>";
            Message msg = MessageFactory.getInstance().getMessage();
            msg.getBody().add(body.getBytes());
            
           
            InputStream inputStream = ClassUtil.getResourceAsStream("MessagePersisterTest.xml", this.getClass());
            ConfigTree config = ConfigTree.fromInputStream(inputStream);

            MessagePersister persister = new MessagePersister(config);
            persister.initialise();
            persister.process(msg);
            
             //now we can check the messageStore to see if our message made it in there.
            System.out.println("Reading the messages in the messageStore");
            
            MessageStore store = MessageStoreFactory.getInstance().getMessageStore();
            Map<URI, Message> messages = store.getAllMessages("TEST");
            messages = store.getAllMessages("TEST");
            Iterator<Message> iter=messages.values().iterator();
            while (iter.hasNext()) {
                Message message=iter.next();
                String bodyFromStore = new String((byte[]) message.getBody().get());
                System.out.println("Body=" + bodyFromStore);
            }
            System.out.println("Getting message for classification TEST:" + messages);
            assertEquals(1, messages.size());
            
            //Now check if it is the same message
            Message message=messages.values().iterator().next();
            String bodyFromStore = new String((byte[]) message.getBody().get());
            assertEquals(body, bodyFromStore);
		}
		catch (Exception re)
		{
			re.printStackTrace();
			assertTrue(false);
		}
	}


	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter(MessagePersisterUnitTest.class);
	}
    
    @BeforeClass
    public static void runBeforeAllTests()
    {
        try {
            File testResourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/test/resources/");
            System.out.println("Current dir=" + testResourceDir.getCanonicalPath());
            DOMConfigurator.configure(testResourceDir.getCanonicalPath() + "/log4j.xml");
            File buildDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/build/");
            File resourceDir = TestEnvironmentUtil.findResourceDirectory("./product/services/jbossesb/src/main/resources/");
            System.setProperty("org.jboss.soa.esb.propertyFile", "jbossesb-unittest-properties.xml");                    
            if ("org.hsqldb.jdbcDriver".equals(Configuration.getStoreDriver())) {
                final String databaseFile = buildDir + "/hsqltestdb" ;
                HsqldbUtil.dropHsqldb(databaseFile);
                server = HsqldbUtil.startHsqldb(databaseFile, "jbossesb");
                
                //Drop what is there now, if exists. We want to start fresh.                
                String sqlCreateCmd    = TestEnvironmentUtil.readTextFile(new File(resourceDir.getCanonicalPath() + "/message-store-sql/hsqldb/create_database.sql"));
                String sqlDropCmd      = TestEnvironmentUtil.readTextFile(new File(resourceDir.getAbsolutePath() + "/message-store-sql/hsqldb/drop_database.sql"));
                
                ConnectionManager mgr = ConnectionManagerFactory.getConnectionManager();
                mgr.init();
                Connection con = mgr.getConnection();
                Statement stmnt = con.createStatement();
                System.out.println("Dropping the schema if exist");
                stmnt.execute(sqlDropCmd);
                System.out.println("Creating the message store schema");
                stmnt.execute(sqlCreateCmd);
            }
        } catch (Throwable e) {
            e.printStackTrace();
            System.out.println("We should stop testing, since we don't have a db.");
            assertTrue(false);
        }
        
    }
    
    @AfterClass
    public static void runAfterAllTests() throws Exception{
        if (Configuration.getStoreDriver().equals("org.hsqldb.jdbcDriver")) {
            HsqldbUtil.stopHsqldb(server);
        }
    }

}
