/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import java.util.List;

import org.jboss.internal.soa.esb.services.rules.RuleInfoBuilder;
import org.jboss.internal.soa.esb.services.rules.RuleServiceCallHelper;
import org.jboss.internal.soa.esb.services.rules.RuleServiceException;
import org.jboss.internal.soa.esb.services.rules.util.RuleConfigUtil;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.services.rules.RuleInfo;

/**
 * BusinessRulesProcessor extends {@link ContentBasedRouter} but does not perform any routing, it only 
 * executes the business rules.
 * <p/>
 * 
 * Configuration Example:
 *<pre>{@code
 *
 *<action class="org.jboss.soa.esb.actions.BusinessRulesProcessor" name="MyAction">
 *    <property name="ruleSet" value="OrderDiscountOnMultipleOrders.drl" />
 *    <property name="ruleReload" value="false" />
 *    <property name="ruleFireMethod" value="FIRE_ALL_RULES" />
 *    <property name="ruleAuditType" value="THREADED_FILE" />
 *    <property name="ruleAuditFile" value="/tmp/event" />
 *    <property name="ruleAuditInterval" value="1000" />
 *    <property name="ruleClockType" value="REALTIME" />
 *    <property name="ruleEventProcessingType" value="STREAM" />
 *    <property name="ruleMultithreadEvaluation" value="false" />
 *    <property name="ruleMaxThreads" value="1" />
 *    <property name="stateful" value="true" />
 *    <property name="defaultContinue" value="trur" />
 *    <property name="object-paths">
 *        <object-path esb="body.TheOrderHeader" />
 *        <object-path esb="body.TheCustomer" />
 *        <object-path esb="body.TheOrderStatus" entry-point="OrderStatusEntry" />
 *        <object-path esb="body.TheOrderInfo" entry-point="OrderInfoEntry" />
 *    </property>
 *    <property name="channels">
 *        <!-- chan1 and chan2 are equivalent (but timeout only applies if async == false) -->
 *        <send-to channel-name="chan1" service-category="cat1" service-name="svc1" />
 *        <send-to channel-name="chan2" channel-class="org.jboss.soa.esb.services.rules.ServiceChannel" service-category="cat1" service-name="svc1" async="true" timeout="30000" set-payload-location="org.jboss.soa.esb.message.defaultEntry" />
 *        <!-- a custom channel -->
 *        <send-to channel-name="chan3" channel-class="com.example.MyChannel" />
 *    </property>
 *</action>
 *}</pre>
 *
 * As this class extends {@link ContentBasedRouter} please see its javadoc for 
 * configuration descriptions that are common to both classes.
 * 
 * Property description:
 * <ul>
 * <li> <i>class</i> action class, org.jboss.soa.esb.actions.BusinessRulesProcessor
 * <li> <i>ruleFireMethod</i> Optional property that defines, for StatefulKnowledgeSessions, if fireAllRules() or fireUntilHalt() should be called.  Possible values are FIRE_ALL_RULES (the default) or FIRE_UNTIL_HALT.
 * <li> <i>ruleAuditType</i> Optional property that defines the type of KnowledgeRuntimeLogger used (CONSOLE, FILE, THREADED_FILE).  If not defined and neither is ruleAuditFile, no auditing is done.  If not defined but ruleAuditFile is, the assumption is THREADED_FILE.
 * <li> <i>ruleAuditFile</i> Optional property that defines the file path for a ruleAuditType of FILE or THREADED_FILE ("event" is the default, and ".log" is always appended).
 * <li> <i>ruleAuditInterval</i> Optional property that defines the interval (in milliseconds) for a ruleAuditType of THREADED_FILE (1000 is the default).
 * <li> <i>ruleClockType</i> Optional property that defines the type of clock used (REALTIME, PSEUDO).  Default is up to drools.
 * <li> <i>ruleEventProcessingType</i> Optional property that defines the type of event processing used (STREAM, CLOUD).  Default is up to drools.
 * <li> <i>ruleMultithreadEvaluation</i> Optional property that defines whether or not to enable KnowledgeBase partitioning. The default is null, which delegates to Drools' default, which is false.
 * <li> <i>ruleMaxThreads</i> Optional property that defines the number of threads to use for KnowledgeBase partitioning. This is only respected if ruleMultithreadEvaluation is true. The default is null, which delegates to Drools' default..
 * <li> <i>stateful</i> Optional property which tells the RuleService to use a stateful session where facts will be remembered between invocations.
 * <li> <i>object-paths</i> Optional property to pass Message objects into Rule Services WorkingMemory.  If an entry-point is specified, that will be placed in the StatefulSession's WorkingMemoryEntryPoint with the same name.
 * <li> <i>channels</i> A set of send-­to properties each containing the logical name 
 * 							of the destination along with the Service category and name 
 *  						as referenced in the registry (if the default channel-class is used),
 *  						OR your own custom-class, which MUST implement org.drools.runtime.Channel,
 *  						, but MAY provide a setConfig(ConfigTree):void method.
 *  						<br/> The logical name is the name which should be used in the rule set.
 * </ul>
 * </br>
 * 
 * @author John Doe
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class BusinessRulesProcessor extends ContentBasedRouter
{
    private RuleServiceCallHelper ruleServiceCallHelper;

	public BusinessRulesProcessor(final ConfigTree config) throws ConfigurationException, RegistryException, MessageRouterException
	{
        super(config);
	    ruleServiceCallHelper = new RuleServiceCallHelper(config);
	}
	
	/** 
     * Inspect the content of the message using a rule set 
     *  
     * @param message
     * @return Message
     * @throws ActionProcessingException
	 */
	public Message process(Message message) throws ActionProcessingException
	{
        try 
        {
            final List<Object> objectList = _mapper.createObjectList(message, _messagePathList);
            return executeRulesService( message, objectList );
        } 
        catch (final ObjectMappingException e) 
        {
            throw new ActionProcessingException( e.getMessage(), e);
        } 
        catch (final RuleServiceException e) 
        {
            throw new ActionProcessingException( e.getMessage(), e);
        }
	}
	
	Message executeRulesService(Message message, final List<Object> objectList) throws RuleServiceException 
	{
		final RuleInfoBuilder builder =  ruleServiceCallHelper.getRuleInfoBuilder();
		builder.global("message", message);
		builder.facts(RuleConfigUtil.extractObjectsFromMessage(entryPointMap, message));
		builder.defaultFacts(objectList);
		final RuleInfo ruleInfo = builder.build();
		return ruleServiceCallHelper.executeRulesService(ruleInfo, message);
	}
	
}
