/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.testutils.FileUtil;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Test the jBPM command interpreter class
 *
 * @author <a href="mailto:kurt.stam@jboss.com">Kurt Stam</a>
 */


public class ContentBasedWiretapUnitTest
{
	static Logger 				_logger = Logger.getLogger(ContentBasedWiretapUnitTest.class);
    static Message message = null;
    
    private static EPR epr1;
    private static EPR epr2;
    private static EPR epr3;
    private static MockCourier courier1;
    private static MockCourier courier2;
    private static MockCourier courier3;
    
    private static ConfigTree[] actions;
	
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(ContentBasedWiretapUnitTest.class);
    }
    
    
    @BeforeClass
    public static void before() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();

        epr1 = new EPR(new URI("test1"));
        epr2 = new EPR(new URI("test2"));
        epr3 = new EPR(new URI("DLS"));
        courier1 = new MockCourier(true);
        courier2 = new MockCourier(true);
        courier3 = new MockCourier(true);

        MockRegistry.register("test", "java", epr1, courier1);
        MockRegistry.register("test", "xml", epr2, courier2);
        MockRegistry.register(ServiceInvoker.INTERNAL_SERVICE_CATEGORY, ServiceInvoker.DEAD_LETTER_SERVICE_NAME, epr3, courier3);
 
        message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
        message.getBody().add(("Hello CBR"));
        
        InputStream in = ClassUtil.getResourceAsStream("ContentBasedWiretapUnitTest.xml", ContentBasedWiretapUnitTest.class);
        String xml = FileUtil.readStream(in);
        actions = ConfigTree.fromXml(xml).getChildren("action");
    }
    
    @Test
	public void wrongDestinationName()
    {
		try {
            ContentBasedWiretap cbr = new ContentBasedWiretap(actions[0]);
            cbr.process(message);
            //the config is wrong so we should error here, no message to the DLS
            assertTrue(false);
		} catch (Exception e) {
            /* As it can't send the message to the DLS the log should say:
               ERROR [main][ContentBasedWiretap] Destination serialized-destination does not exist your configuration
               ERROR [main][ContentBasedWiretap] No rule destination(s) were matched, [serialized, xml]. Please fix your configuration and/or routing rules.
             */
			assertTrue(true);
		}
    }
    
    @Test
    public void correctDestinationName()
    {
        try {
            ContentBasedWiretap cbr = new ContentBasedWiretap(actions[1]);
            Message returnMessage = cbr.process(message);
            //the cbr should return a null message
            assertNotNull(returnMessage);
        } catch (Exception e) {
            assertTrue(false);
        }
    }
    
    @Test
    public void noDestinationMatches()
    {
        try {
            ContentBasedWiretap cbr = new ContentBasedWiretap(actions[2]);
            cbr.process(message);
            //no destination matches, expecting error, and try to send the message to the DLS
            assertTrue(false);
        } catch (Exception e) {
            assertTrue(true);
        }
    }
    
    @Test
    public void extractEntryPointsFromConfig() throws ConfigurationException, RegistryException, MessageRouterException
    {
        final Map<String, List<String>> entryPointsMap = new HashMap<String, List<String>>();
        
        final List<String> paths1 = new ArrayList<String>();
        final String orderPath = "body.Order";
        paths1.add(orderPath);
        
        final List<String> paths2 = new ArrayList<String>();
        final String counterPath = "body.Counter";
        paths2.add(counterPath);
        
        entryPointsMap.put("EntryPoint1", paths1);
        entryPointsMap.put("EntryPoint2", paths2);
        
        final ConfigTree config = new CBRConfigTreeBuilder(true).ruleFile("JBossESBPricingRulesStateful.drl").entryPoints(entryPointsMap).build();
        // Will call checkMyParams which will populate the maps below using the contents from the ConfigTree.
        final ContentBasedRouter contentBasedRouter = new ContentBasedRouter(config);
        
        final Map<String, List<String>> entryPoints = contentBasedRouter.entryPointMap;
        final List<String> entryPoint1 = entryPoints.get("EntryPoint1");
        final List<String> entryPoint2 = entryPoints.get("EntryPoint2");
        
        assertNotNull(entryPoint1);
        assertNotNull(entryPoint2);
        assertEquals(1, entryPoint1.size());
        assertEquals(1, entryPoint2.size());
        assertTrue(entryPoint1.contains(orderPath));
        assertTrue(entryPoint2.contains(counterPath));
        assertFalse(contentBasedRouter._messagePathList.contains(orderPath));
        assertFalse(contentBasedRouter._messagePathList.contains(counterPath));
    }
}