/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.smooks;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.aggregator.AggregateDetails;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.milyn.SmooksException;
import org.milyn.event.report.annotation.VisitBeforeReport;
import org.milyn.event.report.annotation.VisitAfterReport;
import org.milyn.cdr.annotation.ConfigParam;
import org.milyn.container.ExecutionContext;
import org.milyn.delivery.annotation.Initialize;
import org.milyn.delivery.annotation.VisitAfterIf;
import org.milyn.delivery.annotation.VisitBeforeIf;
import org.milyn.delivery.dom.DOMElementVisitor;
import org.milyn.delivery.sax.SAXElement;
import org.milyn.delivery.sax.SAXVisitAfter;
import org.milyn.delivery.sax.SAXVisitBefore;
import org.milyn.javabean.repository.BeanRepository;
import org.w3c.dom.Element;

import java.io.IOException;
import java.io.Serializable;

/**
 * ESB Message Fragment Router.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
@VisitBeforeIf(	condition = "parameters.containsKey('routeBefore') && parameters.routeBefore.value == 'true'")
@VisitAfterIf(	condition = "!parameters.containsKey('routeBefore') || parameters.routeBefore.value != 'true'")
@VisitBeforeReport(summary = "Routing contents of beanId '${resource.parameters.beanId}' to Service '${resource.parameters.serviceCategory}:${resource.parameters.serviceName}'")
@VisitAfterReport(summary = "Routing contents of beanId '${resource.parameters.beanId}' to Service '${resource.parameters.serviceCategory}:${resource.parameters.serviceName}'")
public class FragmentRouter implements DOMElementVisitor, SAXVisitBefore, SAXVisitAfter {

    private static Logger logger = Logger.getLogger(FragmentRouter.class);

    @ConfigParam
    private String beanId;

    @ConfigParam
    private String serviceCategory;

    @ConfigParam
    private String serviceName;

    private ServiceInvoker serviceInvoker;

    @ConfigParam(name = MessagePayloadProxy.SET_PAYLOAD_LOCATION, defaultVal = Body.DEFAULT_LOCATION)
    private String setPayloadLocation;

    @Initialize
    public void intitialize() throws MessageDeliverException {
        serviceInvoker = new ServiceInvoker(serviceCategory, serviceName);
    }

    public void visitBefore(SAXElement element, ExecutionContext executionContext) throws SmooksException, IOException {
        route(executionContext);
    }

    public void visitAfter(SAXElement element, ExecutionContext executionContext) throws SmooksException, IOException {
        route(executionContext);
    }

    public void visitBefore(Element element, ExecutionContext executionContext) throws SmooksException {
        route(executionContext);
    }

    public void visitAfter(Element element, ExecutionContext executionContext) throws SmooksException {
        route(executionContext);
    }

    private void route(ExecutionContext executionContext) {
        Object object = BeanRepository.getInstance(executionContext).getBean(beanId);

        if(object == null) {
            // Try the exec context...
            object = executionContext.getAttribute(beanId);
        }
                
        int currentRecord = 0;
        Integer currentCount = (Integer) executionContext.getAttribute(AggregateDetails.SEQUENCE_COUNT);
        if (currentCount != null) {
        	try {
        		currentRecord = currentCount.intValue();
        	} catch (NumberFormatException nfe) {
        		currentRecord = 0;
        	}
        }
        String seriesUUID = (String) executionContext.getAttribute(AggregateDetails.SERIES_UUID);

        if(object != null) {
            Message message = MessageFactory.getInstance().getMessage();

            if(!(object instanceof Serializable)) {
                logger.warn("Request to route non Serializable Object.  beanId: '" + beanId + "'.  Type: " + object.getClass().getName());
            }

            message.getBody().add(setPayloadLocation, object);
            
            // Set aggregation details for streaming aggregator
            currentRecord++;
            message.getProperties().setProperty(AggregateDetails.AGGREGATE_DETAILS, 
            		new AggregateDetails(seriesUUID, Integer.valueOf(currentRecord))) ;
            executionContext.setAttribute(AggregateDetails.SEQUENCE_COUNT, Integer.valueOf(currentRecord));
            
            try {
                serviceInvoker.deliverAsync(message);
            } catch (MessageDeliverException e) {
                logger.error("Failed to route Object.  beanId: '" + beanId + "'.  Type: " + object.getClass().getName(), e);
            }
        } else {
            logger.info("Request to route unknown Object.  beanId: '" + beanId + "'.");
        }
    }
}